# Copyright (c) 2011, 2013, Oracle and/or its affiliates. All rights reserved. 
#
#   NAME
#      crsutils.pm
#
#   DESCRIPTION
#      This module contains common functions for root scripts
#
#   NOTES
#      <other useful comments, qualifications, etc.>
#
#   MODIFIED   (MM/DD/YY)
#   rdasari     01/11/13 - fix to disable security bug
#   xyuan       01/10/13 - Call srvctl add listener without listener username
#                          if GI user = listener username
#   xyuan       01/04/13 - Fix bug 16072476
#   xyuan       12/26/12 - Fix bug 14768261
#   xyuan       12/20/12 - Fix bug 16026674
#   xyuan       12/20/12 - XbranchMerge xyuan_bug-15969912 from main
#   samjo       12/10/12 - Bug 15980929. Correct list for Linux
#   xyuan       12/09/12 - Fix bug 15969912 - Add listener on the first node
#   xyuan       12/06/12 - Add CVU baseline command
#   xyuan       11/27/12 - Fix bug 14836852
#   rdasari     11/14/12 - modify isODA
#   xyuan       10/29/12 - Fix bug 14796373 & 14827104
#   rdasari     08/22/12 - add isODA()
#   xyuan       10/22/12 - Set the ckpt ROOTCRS_STACK to CKPTFAIL in sub
#                          dietrap
#   xyuan       10/14/12 - Fix bug 14156740 - add sub compact_instlststr
#   xyuan       10/10/12 - Fix bug 14634138
#   rdasari     10/10/12 - add ASM_MODE const
#   spavan      10/09/12 - fix bug14739163 - isCVUConfigured should use old crs
#                          home
#   yizhang     10/08/12 - fix bug 14597440
#   epineda     09/27/12 - Bugfix 13539130
#   spavan      09/26/12 - fix bug14218734 - config cvu with interval parsing
#   shullur     09/25/12 - For moving getcrsrelver to crsutils.pm
#   xyuan       09/18/12 - Fix bug 14621340 - make error in
#                          removing CVU fatal
#   shullur     09/17/12 - For adding the subroutine to check if the version is
#                          greater than or equal to 12.1
#   rdasari     09/13/12 - use GNS_TYPE to configure shared GNS
#   shullur     09/13/12 - For fixing the current working dir in
#                          crf_delete_bdb. Bug 14614307.
#   rtamezd     09/12/12 - Fix bug 14115195
#   rdasari     09/12/12 - remove skipping validation for some parameters
#   rdasari     09/12/12 - add disableRemoteASM()
#   xyuan       09/06/12 - Fix bug 14577135
#   xyuan       09/04/12 - Fix bug 14560280
#   xyuan       08/31/12 - Fix bug 14535011 - Add '-init' option
#   jmunozn     08/24/12 - Remove run_as_user3 and add get_qosctl_path
#   xyuan       08/22/12 - Fix bug 14523800 - Add 'force' for importing ASM
#                          credentials
#   xyuan       08/20/12 - Fix bug 14507349
#   rdasari     08/16/12 - add queryVoteDisks
#   csivanan    08/10/12 - bug/14115152
#   xyuan       08/07/12 - Fix bug 9584563
#   xyuan       08/03/12 - Fix bug 14392140
#   xyuan       08/01/12 - Fix bug 13730408
#   xyuan       07/31/12 - Correct the log file name for prepatch and postpatch
#   xyuan       07/30/12 - Fix bug 14386036
#   rdasari     07/26/12 - change node role if start fails in hub mode
#   xyuan       07/24/12 - Fix bug 14328087
#   rdasari     07/18/12 - add ioservers only for BC
#   rdasari     07/16/12 - set rc before running kfod
#   xyuan       07/12/12 - Fix bug 14302401
#   rdasari     07/10/12 - do not add asm or create diskgroup on client cluster
#   xyuan       07/05/12 - Fix bug 14073012
#   ssprasad    07/03/12 - Fix bug 14272326
#   xyuan       06/21/12 - Fix bug 14223624
#   shullur     06/15/12 - For handling default case of bdb location
#   xyuan       06/12/12 - Fix bug 14117806
#   rtamezd     06/18/12 - Fix bug 14121895
#   rdasari     06/04/12 - modify the .bin file check in set_file_perms
#   xyuan       05/24/12 - Fix bug 14111446
#   xyuan       05/18/12 - Fix bug 14082413 - Add isFirstNodeToConfig
#   tvbabu      05/09/12 - bug 14010695 replacing POSIX tmpnam with tempfile
#   mprasant    05/03/12 - remove REPLICA in 12c
#   rtamezd     05/03/12 - Fix bug 13993142
#   rtamezd     04/25/12 - Removed -subnet option from add_mgmt_db_listener
#   xyuan       04/24/12 - Add options for joining an upgraded cluster
#   sidshank    04/20/12 - fix for bug 13926993.
#   rdasari     04/19/12 - create ioserver before asm listener (bug 13892154)
#   rtamezd     04/15/12 - Fix bug 13953338
#   xyuan       04/13/12 - Fix bug 13954041
#   sidshank    04/09/12 - Removing the code setting ORACLE_OWNER from
#                          environment on Windows.
#   xyuan       04/05/12 - Fix bug 13929451
#   gmaldona    03/30/12 - Move oc4j setup functions to crsoc4j.pm
#   rtamezd     03/28/12 - Fix bug 13899926
#   rtamezd     03/26/12 - Move stop_ohasd_resources from crsinstall.pm
#   rtamezd     03/23/12 - Fix bug 13868122
#   rtamezd     03/22/12 - Fix bug 13807580
#   anutripa    03/12/12 - Disable CRF if MGMTDB not enabled
#   xyuan       03/12/12 - Fix bug 13833629
#   xyuan       03/10/12 - Fix the recursion problem that the trace routine
#                          calls dieformat which in turn calls trace
#   rtamezd     03/09/12 - Fix bug 13822648
#   xyuan       03/08/12 - Fix bug 13797791
#   rtamezd     03/06/12 - Fix bug 13811490
#   rsreekum    03/02/12 - Modify oifcfgNetworks to skip public for dev_env
#   rtamezd     02/23/12 - Fix bug 13768280
#   xyuan       02/23/12 - Fix trace statements
#   rtamezd     02/20/12 - Fix bug 9660844
#   xyuan       02/20/12 - Support new patching interfaces
#   rtamezd     02/17/12 - Fix bugs 13725099 and 13721908
#   sidshank    02/09/12 - remove first_node_tasks subroutine
#   xyuan       02/07/12 - Add getNodeStatus subroutine 
#   rtamezd     02/07/12 - Fix bug 13643262
#   rdasari     02/07/12 - modify isFirstNodeToStart
#   rtamezd     01/26/12 - Fix bug 13621968
#   rtamezd     01/24/12 - Make configFirstNode return at first failure
#   rtamezd     01/19/12 - Abort configure_OCR if OCR is already configured
#   gsanders    01/15/12 - Eliminate ACFS registry
#   rtamezd     01/17/12 - Moved add_localOlr_OlrConfig_OcrConfig from
#                          crsinstall
#   sidshank    01/12/12 - Adding subroutine getCurrentNodenameList()
#   xyuan       01/08/12 - Add utility functions for downgrade
#   shullur     01/06/12 - XbranchMerge shullur_bug-12639016 from st_has_11.2.0
#   shullur     01/06/12 - XbranchMerge shullur_bug-12976590 from st_has_11.2.0
#   rdasari     01/05/12 - modify remove_checkpoints
#   rtamezd     01/05/12 - Fix bug 13560019
#   xyuan       12/26/11 - Fixed bug 13532044
#   spavan      12/22/11 - fix bug13401742
#   xesquive    12/20/11 - add isAutoNode function
#   rtamezd     12/16/11 - Fix bug 13430204
#   sidshank    12/13/11 - Adding platform check for windows for verifying the
#                          executablity of crsctl/srvctl based on uid and gid
#   shullur     12/12/11 - XbranchMerge shullur_bug-12673587 from st_has_11.2.0
#   shullur     11/03/11 - XbranchMerge shullur_bug-12614795 from st_has_11.2.0
#   xyuan       12/08/11 - Fix oifcfgNetworks so that it can deal with IPV6
#   xyuan       12/04/11 - Fix bug 13091719
#   agraves     11/30/11 - Add new constant for REBOOT.
#   rtamezd     11/30/11 - Fix bug 13439445
#   xyuan       11/30/11 - Fix bug 13445306
#   rdasari     11/30/11 - remove isFirstNode parameter
#   anjiwaji    11/29/11 - Add subroutine to lock ACFS files
#   xyuan       11/22/11 - Fix bug 13390751
#   sidshank    11/21/11 - Adding startExclCRS subroutine
#   sidshank    11/15/11 - adding add_mgmt_db_listener subroutine.
#   sowong      11/14/11 - fix bug13109878 - revert oc4j to 11.2 state
#   xyuan       11/14/11 - Fix bug 13340630
#   xyuan       11/05/11 - Fix bug 13329109
#   shullur     11/03/11 - XbranchMerge shullur_bug-12614795 from st_has_11.2.0
#   shullur     11/03/11 - XbranchMerge shullur_bug_11852891 from st_has_11.2.0
#   xyuan       11/02/11 - Fix bug 12945012 & 13251040
#   xyuan       10/31/11 - Fix bug 13328777
#   xyuan       10/25/11 - Fix bug 13006149
#   xyuan       10/10/11 - Support Private and ASM networks
#   xyuan       09/29/11 - Improvements for 12c upgrade
#   rtamezd     09/26/11 - Added new parameters
#   wsun        09/08/11 - Add oc4j as root
#   xyuan       09/07/11 - Fix bug 12908387
#   sidshank    08/11/11 - adding the fail condition for superUser check in
#                          non-dev environment for bug 12739826
#   xyuan       08/09/11 - Far ASM support
#   xyuan       08/04/11 - XbranchMerge xyuan_bug-12795595 from
#                          st_has_11.2.0.3.0
#   xyuan       08/03/11 - XbranchMerge xyuan_bug-12752359 from
#                          st_has_11.2.0.3.0
#   xyuan       08/03/11 - Fix bug 11851866
#   xyuan       07/27/11 - XbranchMerge xyuan_bug-12585291 from
#                          st_has_11.2.0.3.0
#   xyuan       07/27/11 - XbranchMerge xyuan_bug-12727247 from
#                          st_has_11.2.0.3.0
#   xyuan       07/27/11 - XbranchMerge xyuan_bug-12701521 from
#                          st_has_11.2.0.3.0
#   xesquive    07/26/11 - Add functions set_bold and reset_bold
#   xyuan       07/17/11 - BC commands for 12c
#   spavan      06/19/11 - add method to remove cvu
#   rdasari     06/16/11 - add credentials routines
#   xyuan       06/10/11 - expose stack startup levels
#   rdasari     06/02/11 - skips validation for big cluster parameters
#   lmortime    05/25/11 - Making EVMD start first
#   dpham       05/23/11 - Modulize upgrade function
#   rdasari     05/20/11 - set_file_perms fix
#   dpham       03/28/11 - New for 12c
#
package crsutils;

use strict;
use English;
use Exporter;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;
use File::Temp qw/ tempfile/;
use Sys::Hostname;
use Carp;
use Cwd;
use Socket;
use Env qw(NLS_LANG);
use Env qw(SRVM_TRACE);
use Term::ANSIColor;
use Scalar::Util 'blessed';

use constant ERROR                => "-1";
use constant FAILED               => "0";
use constant SUCCESS              => "1";
use constant TRUE                 => "1";
use constant FALSE                => "0";
use constant WARNING              => "2";
use constant REBOOT               => "3";
use constant CKPTSUC              => 'SUCCESS';
use constant CKPTFAIL             => 'FAIL';
use constant CKPTSTART            => 'START';

# How much of the stack do we want to start?
use constant START_STACK_NONE          => 0;
use constant START_STACK_EVMD          => START_STACK_NONE + 1;  # Start EVMD
use constant START_STACK_MDNSD         => START_STACK_EVMD + 1;  # Start MDNSD
use constant START_STACK_GPNPD         => START_STACK_MDNSD + 1; # Start GPNPD
use constant START_STACK_GIPCD         => START_STACK_GPNPD + 1; # Start GIPCD
use constant START_STACK_CSSD          => START_STACK_GIPCD + 1; # Start CSSD & GIPCD
use constant START_STACK_CTSSD         => START_STACK_CSSD + 1;  # Start CTSSD
use constant START_STACK_CRF           => START_STACK_CTSSD + 1; # Start CHM
use constant START_STACK_HAIP          => START_STACK_CRF + 1;   # Start HAIP
use constant START_STACK_ASM           => START_STACK_HAIP + 1;  # Start ASM
use constant START_STACK_CRSD          => START_STACK_ASM + 1;   # Start CRSD
use constant START_STACK_ALL           => START_STACK_CRSD + 1;

# Stuff concerning big cluster
use constant NODE_ROLE_HUB        => 'hub';
use constant NODE_ROLE_RIM        => 'leaf';
use constant NODE_ROLE_AUTO       => 'auto';

# standard and flex/remote asm modes
use constant ASM_MODE_LEGACY      => 'legacy';
use constant ASM_MODE_FLEX        => 'remote';

use constant GI_STACK_DOWN        => '0';
use constant GI_STACK_UP          => '1';
use constant GI_STACK_PARTIAL     => '2';

# Becase oracss.pm uses the constants defined here, export in a BEGIN
# block so that they will not cause a compilation failure
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
our $CFG;

BEGIN {
   @ISA = qw(Exporter);
   
   my @exp_const = qw(ERROR FAILED SUCCESS WARNING TRUE FALSE REBOOT CKPTSUC CKPTFAIL CKPTSTART
                      START_STACK_NONE START_STACK_EVMD START_STACK_MDNSD START_STACK_GPNPD
                      START_STACK_GIPCD START_STACK_CRF START_STACK_CTSSD
                      START_STACK_CSSD START_STACK_ASM START_STACK_HAIP
                      START_STACK_CRSD START_STACK_ALL
                      GI_STACK_DOWN GI_STACK_PARTIAL GI_STACK_UP 
                      NODE_ROLE_HUB NODE_ROLE_RIM 
                      ASM_MODE_LEGACY ASM_MODE_FLEX
                     );

   my @exp_func  = qw(trace trim backtrace error dietrap dieformat is_dev_env 
                      tolower_host run_as_user run_as_user2 check_SuperUser 
                      copy_file create_dir create_dirs crs_exec_path run_crs_cmd
                      system system_cmd system_cmd_capture read_file print_error
                      validate_ocrconfig validate_olrconfig upgrade_local_OCR
                      get_ocrdisk get_ocrloc3disk get_ocrloc4disk get_ocrloc5disk
                      get_ocrlocaldisk get_ocrmirrordisk start_resource srvctl
                      extractVotedisks configure_OCR start_service check_service
                      oifcfgiflst_to_instlststr isCVUConfigured setperm_vdisks
                      wait_for_stack_start create_OLR_keys isResRunning srvctl_tty
                      instantiate_scripts copy_wrapper_scripts set_file_perms
                      isCkptexist isCkptSuccess getCkptStatus isCkptPropertyExists
                      writeCkpt writeCkptPropertyFile writeCkptProperty getVerInfo
                      getCkptPropertyValue check_dir check_file checkServiceDown
                      quoteDiskGroup add_Nodeapps start_Nodeapps get_crs_version 
                      stop_resource set_perms_ocr_vdisk add_ons isReusedg
                      crf_config_generate crf_delete_bdb start_CVU enable_HAIP
                      isPathonASM isRolling isHAIPsupported isOCRonASM isAddNode
                      isACFSSupported isCRFSupported isVersion112 isVersion111 
                      isVersion10 isFirstNodeToStart isRAC_appropriate copydir
                      isPrereqIgnored stop_diskmon ipv4_atol set_logging
                      validate_ocrconfig configFirstNode oifcfg_intf_parse add_CVU
                      remove_checkpoints stopClusterware isOwnerGroupValid
                      isRolling start_clusterware checkOCR unlockHAHomefordeinstall
                      validateOCR validate_SICSS isONSexist setNodeRole
                      system_cmd_capture_noprint unlockCRSHomefordeinstall 
                      isBigCluster setCurrentNodeRole setHubSize isHubNode isRimNode
                      isNearASM createASMCredentials importASMCredentials
                      isValidVersion versionComparison rscPreChecks isASMExists
                      isLegacyASM isFarASM copyASMCredentials get_ons_port
                      remove_CVU set_bold reset_bold isOldVersionLT112 add_ASM
                      createCredDomain setConfiguredCRSHome getConfiguredCRSHome
                      oifcfg_intf_to_instlststr add_GNS add_scan add_scan_listener
                      start_GNS start_scan start_scan_listener getActiveNodeRoles
                      startFullStack stopFullStack bounce_ohasd collect_cvu_baseline
                      startOhasdOnly createMgmtdbDir startExclCRS stopOracleRestart
                      add_mgmt_db_listener getCHMAttrib removeCHMDB movedir isAutoNode
                      get_olsnodes_info getNodeNumList getLocalNodeName isASMConfigured
                      isVersionLT11202 getCurrentNodenameList extractDiskgroups
                      removeVotingDiks addVotingDisks removeVotingfiles addVotingFiles
                      start_crsd_and_check start_excl_crs stop_crsd_and_check
                      startExclNoCRS add_localOlr_OlrConfig_OcrConfig lockAcfsFiles
                      waitForVipRes upgradeModelResType isVersionMatch getNodeStatus
                      checkGIStack unlockHome modifyparamfile check_NewCrsStack
                      stop_ohasd_resources isFirstNodeToConfig isReposBDB
                      print_lines trace_lines print_trace_lines isVersionGTE1201 getcrsrelver
                      disableRemoteASM get_CVU_checkInterval disable_CVU enable_CVU
                      compact_instlststr isODA get_qosctl_path createNetLsnrWithUsername stopNetLsnr
                      startNetLsnr getLsnrUsername isLastNodeToConfig removeNetLsnr
                      getCurrentNetLsnrs upgradeNetLsnrWithUsername isOldVersionLT111
                      startListeners
                     );

   @EXPORT  = qw($CFG);
   push @EXPORT, @exp_const, @exp_func;
}

# root script modules
use crsohasd;
use crsgpnp;
use oracss;
use oraacfs;
use crsoc4j;
use s_crsutils;

# FIXME: These should be moved to crsdelete.pm, which is the place
# they are referenced
my %elements = ('params'              => 'HASH',
                'srvctl_trc_suff'     => 'COUNTER',
                'OLD_CRS_HOME'        => 'INDIRECT', # see %indirect
                'oldcrshome'          => 'INDIRECT', # see %indirect
                'oldcrsver'           => 'INDIRECT', # see %indirect
                'oldconfig'           => 'HASH',
                'CLSCFG_EXTRA_PARMS'  => 'ARRAY',
                'old_nodevips'        => 'ARRAY',
                'hahome'              => 'SCALAR',
                'isRerun'             => 'SCALAR',
                'wipCkptName'         => 'SCALAR',
                'isNodeCrashed'       => 'SCALAR',
                'restart_css'         => 'SCALAR'
               );

# The %indirect hash allows the proper methods to be called from 'new',
# when the initializing hash that is passed to 'new' is not properly
# done.
# The correct way of specifying, for example, 'oldcrsver' and
# 'oldcrshome' in the initializing hash is not:
#    oldcrsver  => $oldcrsver
#    oldcrshome => $oldcrshome
# but is:
#    my @oldcrsversion = split('\.', $oldcrsver);
#    oldconfig  => 
#      {ORA_CRS_VERSION => \@oldcrsversion,
#       ORA_CRS_HOME    => $oldcrshome
#      }
my %indirect = ( 'oldcrsver'    => \&oldcrsver,
                 'oldcrshome'   => \&oldcrshome,
                 'OLD_CRS_HOME' => \&OLD_CRS_HOME
               );

=head2 new

  This is the class constructor method for this class

=head3 Parameters

  A hash containing values for any key that is listed in the accessor
   methods section

=head3 Returns

  A blessed class

=head3 Usage

  my $cfg = crsutils->new(
                paramfile           => $PARAM_FILE,
                OSDFILE             => $defsfile,
                crscfg_trace        => TRUE,
                HOST                => $HOST
                )

  This creates an object with parameters built from $PARAM_FILE
  and $defsfile, for HOST $HOST with tracing turned on.  The values
  specified may be retrieved via the standard access methods, e.g.
    my $host = $cfg->HOST;
  will set $host to the $HOST value set in the hash passed to 'new'.

  While it is possible to pass any key/value pair to 'new', even ones
  for which there are no access methods, the values cannot be easily
  used without an access method.  For the list of access methods, see
  the 'Access Methods Section' below

=cut

# Class constructor and methods
sub new {
   my ($class, %init) = @_;
   $CFG = {};

   for my $element (keys %init) {
      if (defined($init{$element})) {
         my $type = $elements{$element};
         if (($type eq 'ARRAY' || $type eq 'HASH') &&
            ref($init{$element}) ne $type)
         {
            croak "Initializer for $element must be $type reference";
         }

         if ($type ne 'INDIRECT') { $CFG->{$element} = $init{$element}; }
      }
   }

   # Initialize stuff not in the initializer
   for my $element (keys %elements) {
      if (!defined($CFG->{$element})) {
         my $type = $elements{$element};
         if ($type eq 'ARRAY')      { $CFG->{$element} = []; }
         elsif ($type eq 'HASH')    { $CFG->{$element} = {}; }
         elsif ($type eq 'COUNTER') { $CFG->{$element} = 0; }
      }
   }

   bless $CFG, $class;

   for my $element (keys %init) {
      if (defined($init{$element})) {
         my $type = $elements{$element};
         if ($type eq 'INDIRECT') { $indirect{$element}($CFG, $init{$element}); }
      }
   }

   if (!$CFG->HOST) { $CFG->HOST(tolower_host()); }
  
   # The default is we start the entire stack
   if (!$CFG->stackStartLevel) { $CFG->stackStartLevel(START_STACK_ALL); }

   if (! $CFG->paramfile) { die("No configuration parameter file was specified"); }

   if (! -e $CFG->paramfile) {
      die("Configuration parameter file ", $CFG->paramfile,
          " cannot be found");
   }

   print("Using configuration parameter file: ", $CFG->paramfile, "\n");

   # Set up the parameters
   setup_param_vars($CFG->paramfile);

   # Now set various defaults/values based on various input
   my $OH = $CFG->params('ORACLE_HOME'); # for convenience

   if (!$CFG->ORA_CRS_HOME) {
      $CFG->ORA_CRS_HOME($OH);
   }

   if ($OH) {
      trace("Using Oracle CRS home $OH");
   }
   else {
      die("The Oracle CRS home path not found in the configuration parameters");
   }

   if (!(-d $OH)) { die("The Oracle CRS home path \"$OH\" does not exist"); }

   my $default_trc_dir = catfile($OH, 'cfgtoollogs', 'crsconfig');
   my $default_olr_dir = catfile($OH, 'cdata');

   #Don't use a time stamp in a dev environment
   my $timestamp;
   if(is_dev_env())
   {
      $timestamp = "";
   }
   else
   {
      $timestamp = "_" . genTimeStamp();
   }

   if ($CFG->SIHA) {
      # Define stuff for SIHA
      # $CFG->parameters_valid($CFG->validateSIHAVarList);
      # list of params that MUST be specified in the param file
      my @required = ("ORACLE_HOME", "ORACLE_OWNER", "ASM_UPGRADE");
      my $myplatformfamily = s_get_platform_family();
      $myplatformfamily =~ tr/A-Z/a-z/;

      if ($myplatformfamily eq "unix") {
         push @required, "ORA_DBA_GROUP";
      }
    
      # trace file
      if (!$CFG->crscfg_trace_file) {
         my $file = "roothas$timestamp.log";
         if ($CFG->CRSDeconfig) { $file = "hadeconfig.log"; }
         if (($CFG->HAPatch) || ($CFG->PREPATCH) || ($CFG->POSTPATCH))
         { 
           $file = "hapatch$timestamp.log";
         }

         $CFG->crscfg_trace_file(catfile($default_trc_dir, $file));
      }

      $CFG->parameters_valid(validateVarList(@required));

      if (!$CFG->OLR_DIRECTORY) {
         $CFG->OLR_DIRECTORY(catfile($default_olr_dir, 'localhost'));
      }
   }
   else {
      # Define stuff for clustered mode
      if (!$CFG->crscfg_trace_file) {
         my $host = $CFG->HOST;
         my $file = "rootcrs_$host$timestamp.log";
         if ($CFG->CRSDeconfig) { $file = "crsdeconfig_$host.log"; }
         if (($CFG->CRSPatch) || ($CFG->PREPATCH) || ($CFG->POSTPATCH))
         {
           $file = "crspatch_$host$timestamp.log"; 
         }
         if ($CFG->DOWNGRADE) { $file = "crsdowngrade_$host$timestamp.log"; }

         $CFG->crscfg_trace_file(catfile($default_trc_dir, $file));
      }

      $CFG->parameters_valid(validateVarList());

      if (!$CFG->OLR_DIRECTORY) {
         $CFG->OLR_DIRECTORY($default_olr_dir);
      }

      # Do logical validation
      validateLogical();
   }

   # We really should destroy $CFG here; this will be impelmented later
   if ($CFG->parameters_valid) {
      trace("The configuration parameter file", $CFG->paramfile, " is valid");
   }
   else {
      die("The configuration parameter file", $CFG->paramfile, " is not valid");
   }

   if ($CFG->osdfile && -e $CFG->osdfile) {
      setup_param_vars($CFG->osdfile);
   }

   # pull all parameters defined in crsconfig_addparams
   my $addfile = catfile($OH, 'crs', 'install', 'crsconfig_addparams');
   if (-e $CFG->addfile) {
      setup_param_vars($CFG->addfile);
      if ($CFG->defined_param('CRS_ADDNODE') &&
          $CFG->params('CRS_ADDNODE') eq "true")
      {
         $CFG->addnode(TRUE);
      }
      else {
         $CFG->addnode(FALSE);
      }
   }

   my $SUPERUSER = check_SuperUser();

   my $platform  = s_get_platform_family();
   if ($platform ne 'windows') {
      $CFG->HAS_USER($SUPERUSER);
   }

   $CFG->SUPERUSER($SUPERUSER);

   if ($SUPERUSER) {
      $CFG->user_is_superuser(TRUE);
   }
   else {
      # If we are not Superuser and nor is it Dev Environment, fail!!       
      if (! is_dev_env ()) {
         print_error(1);
         exit 2;
      }

      # If we are not SUPERUSER, indicate this and set SUPERUSER to
      # ORACLE_OWNER
      $CFG->user_is_superuser(FALSE);
      $CFG->SUPERUSER($CFG->params('ORACLE_OWNER'));
   }

   # Set some default values, if necessary
   $CFG->OLR_LOCATION(catfile($CFG->OLR_DIRECTORY, $CFG->HOST . '.olr'));

   if ($CFG->SUPERUSER &&
       $CFG->defined_param('OLASTGASPDIR') &&
       ! -e $CFG->params('OLASTGASPDIR'))
   {
      mkpath($CFG->params('OLASTGASPDIR'));
   }

   if ((!$CFG->HAS_USER) || ($CFG->HAS_USER && $CFG->SIHA))
   {
      $CFG->HAS_USER($CFG->params('ORACLE_OWNER'));
   }

   if (!$CFG->HAS_GROUP) {
      $CFG->HAS_GROUP($CFG->params('ORA_DBA_GROUP'));
   }

   if (!$CFG->s_run_as_user2p) {
      $CFG->s_run_as_user2p(\&crsutils::s_run_as_user2);
   }

   if (!$CFG->s_run_as_userp) {
      $CFG->s_run_as_userp(\&crsutils::s_run_as_user);
   }

   # If the versions of the 'run_as user' commands with parm order
   # reversed exist, as inidicated in the sybol table, set them now
   if (!$CFG->s_run_as_user2_v2p &&
      defined($s_crsutils::{'s_run_as_user2_v2'})) {
      $CFG->s_run_as_user2p_v2(\&crsutils::s_run_as_user2_v2);
   }

   if (!$CFG->s_run_as_user_v2p &&
      defined($s_crsutils::{'s_run_as_user_v2'}))
   {
      $CFG->s_run_as_userp_v2(\&crsutils::s_run_as_user_v2);
   }

   if (!$CFG->s_get_qosctl_path_p) {
       $CFG->s_get_qosctl_path_p(\&crsutils::s_get_qosctl_path);
   }

   $CFG->platform_family(lc(s_get_platform_family()));

   $CFG->SUCC_REBOOT(0);

   $CFG->print_config;

   # set owner & permission of trace file
   s_set_ownergroup ($CFG->params('ORACLE_OWNER'),
                     $CFG->params('ORA_DBA_GROUP'),
                     $CFG->crscfg_trace_file);
   s_set_perms ("0775", $CFG->crscfg_trace_file);

   # set environment variables
   $ENV{'ORA_CRS_HOME'} = $OH;
   $ENV{'ORACLE_HOME'}  = $OH;
   $ENV{'ORACLE_BASE'}  = $CFG->params('ORACLE_BASE');

   setConfiguredCRSHome(); 

   my $diskgroup;
   if ($CFG->defined_param('ASM_DISK_GROUP'))
   {
      $diskgroup = $CFG->params('ASM_DISK_GROUP');
   }

   if (($CFG->ASM_STORAGE_USED) && $diskgroup)
   {
     $diskgroup =~ s/\$/\\\$/g;
     trace("Save the ASM password file location: +$diskgroup/orapwASM");
     $CFG->ASM_PWD_FILE("+$diskgroup/orapwASM");
   }

   return $CFG;
}

# Access Methods Section
#   This section contains the accessor method used in this class
#
# Adding a new accessor method:
#   Unless the accessor method must do something special, use the
##   standard access methods:
#     access_array  - get/set a value from/in an array
#     access_hash   - get/set a value from/in an hash
#     access_scalar - get/set a scalar value
#
# Examples:
#   (scalar FOOS)
#     (method definition) sub FOOS {return access_scalar(@_);}
#     (value set)   $CFG->FOOS('BARS'); (sets $CFG->FOOS to 'BARS')
#     (value get)   my $foos = $CFG->FOOS;
#
#   (array FOOA)
#     (method definition) sub FOOA {return access_array(@_);}
#     (value set)   $CFG->FOOA(0, 'BARA'); (sets FOOA[0])
#     (value set)   $CFG->FOOA(\@BARA); (sets FOOA array to @BARA)
#     (value get)   my $fooa0 = $CFG->FOOA(0); (gets FOOA[0])
#     (value get)   my @fooa = @{$CFG->FOOA}; (gets @FOOA)
#
#   (hash FOOH)
#     (method definition) sub FOOH {return access_hash(@_);}
#     (value set)   $CFG->FOOH('BARk', 'BARv'); (sets FOOH key BARk to BARv)
#     (value set)   $CFG->FOOH(\%BARH); (sets $CFG->FOOH to %BARH)
#     (value get)   my $barv = $CFG->FOOH('BARk'); (gets FOOH{'BARk'})
#     (value get)   my %fooh = %{$CFG->FOOH}; (gets %FOOH)
#
#   (counter FOOC) initial value preset to 0
#     (method definitions)
#        sub FOOC {return access_counter(@_);}
#        sub pp_FOOC {return access_counter(@_);}
#        sub FOOC_pp {return access_counter(@_);}
#     (value get - assuming a current value of 3)
#        (no value change)
#          my $cval = $CFG->FOOC; (returns 3, current value still 3)
#        (increment before returning ++FOOC)
#          my $cval = $CFG->pp_FOOC; (returns 4, current value now 4)
#        (increment after returning FOOC++)
#          my $cval = $CFG->FOOC_pp; (returns 3, current value now 4)


# Accessor methods for class elements
sub addnode             { return access_scalar(@_); }
sub old_nodevips        { return access_array(@_); }
sub CLSCFG_EXTRA_PARMS  { return access_array(@_); }
sub CLSCFG_POST_CMD     { return access_array(@_); }
sub CRSCFG_POST_CMD     { return access_array(@_); }
sub CRSDeconfig         { return access_scalar(@_); }
sub DEBUG               { return access_scalar(@_); }
sub HAS_GROUP           { return access_scalar(@_); }
sub HAS_USER            { return access_scalar(@_); }
sub HOST                { return access_scalar(@_); }
sub SIHA                { return access_scalar(@_); }
sub OCR_ID              { return access_scalar(@_); }
sub ORA_CRS_HOME        { return access_scalar(@_); }
sub SUPERUSER           { return access_scalar(@_); }
sub VF_DISCOVERY_STRING { return access_scalar(@_); }
sub OLR_DIRECTORY       { return access_scalar(@_); }
sub OLR_LOCATION        { return access_scalar(@_); }
sub UPGRADE             { return access_scalar(@_); }
sub DOWNGRADE           { return access_scalar(@_); }
sub NETWORKS            { return access_scalar(@_); }
sub gpnp_setup_type     { return access_scalar(@_); }
sub addfile             { return access_scalar(@_); }
sub hosts               { return access_array(@_); }
sub osdfile             { return access_scalar(@_); }
sub paramfile           { return access_scalar(@_); }
sub parameters_valid    { return access_scalar(@_); }
sub platform_family     { return access_scalar(@_); }
sub user_is_superuser   { return access_scalar(@_); }
sub oldconfig           { return access_hash(@_); }
sub s_run_as_user2p     { return access_scalar(@_); }
sub s_run_as_user2_v2p  { return access_scalar(@_); }
sub s_run_as_userp      { return access_scalar(@_); }
sub s_run_as_user_v2p   { return access_scalar(@_); }
sub hahome              { return access_scalar(@_); }
sub CRSPatch            { return access_scalar(@_); }
sub HAPatch             { return access_scalar(@_); }
sub FORCE               { return access_scalar(@_); }
sub LASTNODE            { return access_scalar(@_); }
sub REMOTENODE          { return access_scalar(@_); }
sub UNLOCK              { return access_scalar(@_); }
sub UNLOCK_HOME         { return access_scalar(@_); }
sub NOCRSSTOP           { return access_scalar(@_); }
sub isRerun             { return access_scalar(@_); }
sub wipCkptName         { return access_scalar(@_); }
sub isNodeCrashed       { return access_scalar(@_); }
sub restart_css         { return access_scalar(@_); }
sub stackStartLevel     { return access_scalar(@_); }
sub SUCC_REBOOT         { return access_scalar(@_); }
sub DEINSTALL           { return access_scalar(@_); }
sub KEEPDG              { return access_scalar(@_); }
sub ASM_PWD_FILE        { return access_scalar(@_); }
sub configuredCRSHome   { return access_scalar(@_); } 
sub LOCK_ACFS           { return access_scalar(@_); }
sub PREPATCH            { return access_scalar(@_); }
sub POSTPATCH           { return access_scalar(@_); }
sub NONROLLING          { return access_scalar(@_); }
sub DESTCRSHOME         { return access_scalar(@_); }
sub NORESTART           { return access_scalar(@_); }
sub AUTO                { return access_scalar(@_); }
sub JOIN                { return access_scalar(@_); }
sub EXISTINGNODE        { return access_scalar(@_); }
sub isFirstNodeToUpgrade { return access_scalar(@_); }
sub isLastNodeToUpgrade { return access_scalar(@_); }
sub OLD_CSSD_LOG_LEVEL  { return access_scalar(@_); }
sub init                { return access_scalar(@_); }
sub s_get_qosctl_path_p { return access_scalar(@_); }

# Counters
# pp_ for increment before, eg pp_foo same as ++foo
# _pp for increment after, eg foo_pp same as foo++
# both take an argument for increment amount (default increment is 1)
sub srvctl_trc_suff     { return access_counter(@_); }
sub pp_srvctl_trc_suff  { return access_counter(@_); } # ++srvctl_trc_suff
sub srvctl_trc_suff_pp  { return access_counter(@_); } # srvctl_trc_suff++

sub config_value { return $_[0]->{$_[1]}; }

# The following methods are required to translate values associated
# with the old CRS installation that were passed as scalars into
# key/value pairs in the oldconfig hash
# These should not be required, but because code was built around them
# after an incorrect implementation, it was less destabilizing to
# provide translation methods than to correct the code
sub oldcrshome          {
  my $class = shift;
  return oldconfig($class, 'ORA_CRS_HOME', @_);
}

sub OLD_CRS_HOME        { return oldcrshome(@_); }

sub oldcrsver           {
  my $class = shift;
  my $rv;
  my @ver;
  if (scalar(@_) == 0) {
    @ver = @{oldconfig($class, 'ORA_CRS_VERSION')};
    $rv = join('.', @ver);
  }
  else {
    @ver =(split('\.', $_[0]));
    $rv = oldconfig($class, 'ORA_CRS_VERSION', \@ver);
  }
  return $rv;
}

sub params {
  my $cfg = shift;

  # If the parameter has not been defined, error.  This prevents
  # typos from going unnoticed
  if (scalar(@_) == 1 && !defined($cfg->{'params'}->{$_[0]})) {
    die(dieformat(300, $_[0]));
  }
  else { return access_hash($cfg, @_); }
}

sub defined_param {
  my $cfg = shift;
  my $defd = FALSE;
  if (scalar(@_) > 1) { croak "Only 1 parameter allowed: @_"; }
  else { $defd = defined($cfg->{'params'}->{$_[0]}); }

  return $defd;
}

sub defined_value {
  my $cfg = shift;
  my $defd = FALSE;
  if (scalar(@_) > 1) { croak "Only 1 parameter allowed: @_"; }
  else { $defd = defined($cfg->{$_[0]}); }

  return $defd;
}

sub ASM_STORAGE_USED {
  my $cfg = shift;
  my $val;
  my $ret;
  if (@_) { # Setting the value, so keep params in sync
    $ret = shift;
    if ($ret) { $cfg->params('CRS_STORAGE_OPTION', 1); }
    else { $cfg->params('CRS_STORAGE_OPTION', 2); }
  }
  elsif ($cfg->params('CRS_STORAGE_OPTION') == 1) { $ret = TRUE; }
  else { $ret = FALSE; }

  return $ret;
}

# If tracing is turned on after initialization, make sure directory is
# created
sub crscfg_trace {
  my $ret = access_scalar(@_);
  # if the call was to turn tracing on, make sure trace dir created
  if ($ret && scalar(@_) > 1) { setup_trace_dir(); }
  return $ret;
}

sub crscfg_trace_file {
  my $ret = access_scalar(@_);
  # if the call was set the file name, make sure trace dir created
  if (scalar(@_) > 1) { setup_trace_dir(); }
  return $ret;
}

# Low level access methods
sub access_scalar {
  my $class  = shift;

  # find where we were called from so that we know what element
  # Get callers name
  my @caller = caller(1);

  my $name = $caller[3];
  # strip class name to get element name
  my $class_name = ref($class);
  $name =~ s/$class_name\:\://;

  my $ret;
  if (@_ > 1) { croak "Too many args to $name"; }
  if (@_) {$ret = shift; $class->{$name} = $ret; }
  else { $ret = $class->{$name}; }

  return $ret;
}

sub access_array {
  my $class  = shift;

  # find where we were called from so that we know what element
  # Get callers name
  my @caller = caller(1);

  my $name = $caller[3];
  # strip class name to get element name
  my $class_name = ref($class);
  $name =~ s/$class_name\:\://;

  my $init;
  my $ret;

  if (! @_) { $ret = $class->{$name}; }
  else {
    $init = shift;
    if (ref($init) eq 'ARRAY' && !@_) {
      $class->{$name} = $init;
      $ret = $class->{$name};
    }
    elsif (@_ > 1) { croak "Too many args to $name"; }
    elsif (@_) { $class->{$name}->[$init] = $ret = shift; }
    else { $ret = $class->{$name}->[$init]; }
  }

  return $ret;
}

sub access_hash {
  my $class  = shift;

  # find where we were called from so that we know what element
  # Get callers name
  my @caller = caller(1);

  my $name = $caller[3];
  # strip class name to get element name
  my $class_name = ref($class);
  $name =~ s/$class_name\:\://;

  my $init;
  my $ret;
  if (! @_) { $ret = $class->{$name}; }
  else {
    $init = shift;
    if (ref($init) eq 'HASH' && !@_) {
      $class->{$name} = $init;
      $ret = $class;
    }
    elsif (@_ > 1) { croak "Too many args to $name"; }
    elsif (@_) { $class->{$name}->{$init} = $ret = shift; }
    else { $ret = $class->{$name}->{$init}; }
  }

  return $ret;
}

sub access_counter {
  my $class  = shift;

  # find where we were called from so that we know what element
  # Get callers name
  my @caller = caller(1);

  my $name = $caller[3];
  # strip class name to get element name
  my $class_name = ref($class);
  $name =~ s/$class_name\:\://;

  my $elt_name = $name;
  my $pre  = $elt_name =~ s/^pp_//;
  my $post = $elt_name =~ s/_pp$//;

  my $ret = $class->{$elt_name};
  my $incr = 1;
  if (@_ > 1) { croak "Too many args to $name: @_"; }

  if (@_) { $incr = shift; }
  if ($pre) { $class->$elt_name(($ret += $incr)); }
  elsif ($post) { $class->$elt_name($ret + $incr); }

  return $ret;
}

###### End Access Methods Section #######

# Set up the trace directory
sub setup_trace_dir {
  if ($CFG && $CFG->crscfg_trace &&
      $CFG->crscfg_trace_file &&
      ! -e $CFG->crscfg_trace_file) {
    my $trace_dir = dirname($CFG->crscfg_trace_file);
    if (! -e $trace_dir) {
      my $tracing = $CFG->crscfg_trace;

      # temporarily turn off tracing to avoid recursing in create_dir
      $CFG->crscfg_trace(0);
      create_dir($trace_dir);
      $CFG->crscfg_trace($tracing);
    }
  }
}

###---------------------------------------------------------
#### Function for tracing logging messages for root scripts
# ARGS : 0
sub trace
{
    my ($sec, $min, $hour, $day, $month, $year) =
        (localtime) [0, 1, 2, 3, 4, 5];
    $month = $month + 1;
    $year = $year + 1900;

    if ($CFG && $CFG->crscfg_trace) {
      my $CRSCFG_TRACE_FILE = $CFG->crscfg_trace_file;
      if ($CRSCFG_TRACE_FILE) {
        open (TRCFILE, ">>$CRSCFG_TRACE_FILE")
          or die("Cannot open $CRSCFG_TRACE_FILE for appending\n");
      }
      printf TRCFILE  "%04d-%02d-%02d %02d:%02d:%02d: @_\n",
        $year, $month, $day, $hour, $min, $sec;
      close (TRCFILE);
    } else {
      printf "%04d-%02d-%02d %02d:%02d:%02d: @_\n",
        $year, $month, $day, $hour, $min, $sec;
    }
}

####---------------------------------------------------------
#### Function for dumping errors on STDOUT
# ARGS : 0 
sub error
{
    print "@_\n";

    if ($CFG && $CFG->crscfg_trace && $CFG->crscfg_trace_file) {
        trace (@_);
    }
    if ($CFG && $CFG->DEBUG) {
      trace("###### Begin Error Stack Trace ######");
      backtrace();
      trace("####### End Error Stack Trace #######\n");
    }
}

####---------------------------------------------------------
#### Function for getting this host name in lower case with no domain name
# ARGS : 0
sub tolower_host
{
    my $host = hostname () or return "";

    # If the hostname is an IP address, let hostname remain as IP address
    # Else, strip off domain name in case /bin/hostname returns FQDN
    # hostname
    my $shorthost;
    if ($host =~ /^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+$/) {
        $shorthost = $host;
    } else {
        ($shorthost,) = split (/\./, $host);
    }

    # convert to lower case
    $shorthost =~ tr/A-Z/a-z/;

    die(dieformat(308)) if ($shorthost eq "");

    return $shorthost;
}

# Thin wrapper of OSD function to run a command as a specified user
# Parameters:
#   1. user to run command as
#   remaining arguments are the command to run
sub run_as_user
{
  my $user = shift;
  trace("Running as user $user: @_");
  return $CFG->s_run_as_usere($user, @_);
}

# Execute a command as a user (do not invoke directly, use run_as_user)
sub s_run_as_usere {
  my $cfg = shift;
  my $user = shift;
  my $pgm = $cfg->s_run_as_userp;
  my @args = ("@_", $user);
  if ($cfg->s_run_as_user_v2p) {
    $pgm = $cfg->s_run_as_user_v2p;
    @args = ($user, @_);
  }
  return &$pgm(@args);
}

# Thin wrapper of OSD function to run a command as a specified user
# Parameters:
#   1. user to run command as
#   2. array reference for output capture
#   remaining arguments are the command to run
sub run_as_user2
{
  my $user = shift;
  my $aref = shift;
  trace("Running as user $user: @_");
  return $CFG->s_run_as_user2e($user, $aref, @_);
}

# Execute a command as a user, returning the output
# (do not invoke directly, use run_as_user2)
sub s_run_as_user2e {
  my $cfg = shift;
  my $user = shift;
  my $aref = shift;
  my $rc;
  my @args = (\@_, $user, $aref);
  if (!$cfg->s_run_as_user2_v2p) {
    my $pgm  = $cfg->s_run_as_user2p;
    $rc = &$pgm(\@_, $user, $aref);
  }
  else {
    my $pgm = $cfg->s_run_as_user2_v2p;
    my @out = &$pgm($user, @_);
    $rc = shift @out;
    @{$aref} = @out;
  }
  return $rc;
}

sub trim
################################################################################
# Function: Remove leading and trailing blanks.
#
# Arg     : string
#
# Return  : trimmed string
################################################################################
{
   my $str = shift;
   $str =~ s/^\s+//;
   $str =~ s/\s+$//;
   return $str ;
}

####---------------------------------------------------------
#### Function for checking and returning Super User name
# ARGS : 0
sub check_SuperUser
{
    my $superuser = s_check_SuperUser ()
        or trace("Not running as authorized user");
    return $superuser;
}

sub is_dev_env
{
    my $isDevEnv = uc($ENV{'_SCLS_DEVELOPMENT'});
    if ($isDevEnv eq "TRUE") {
        return TRUE;
    } else {
        return FALSE;
    }
}

sub backtrace {
  my $levels = $_[0];
  my $done = FALSE;

  trace(sprintf("    %-15s %-20s %-4s %-10s", "Package", "File",
                "Line", "Calling"));
  trace(sprintf("    %-15s %-20s %-4s %-10s", "-" x 15, "-" x 20,
                "-" x 4, "-" x 10));
  for (my $bt = 1; ((!$levels && !$done) || $bt <= $levels); $bt++) {
    my @caller = caller($bt);
    if (scalar(@caller) == 0) { $done = TRUE; }
    else {
      my $pkg = $caller[0];
      my $file = basename($caller[1]);
      my $line = $caller[2];
      my $sub  = $caller[3];
      trace(sprintf("%2d: %-15s %-20s %4d %s", $bt, $pkg, $file,
                    $line, $sub));
    }
  }
}

sub dietrap {
  trace("###### Begin DIE Stack Trace ######");
  backtrace(0);
  trace("####### End DIE Stack Trace #######\n");
  #wipCkptName is the global variable to store active checkpoint in progress.
  #Needed to handle ctrl-c cases.
  if ($CFG) {
    trace ($CFG->wipCkptName . " checkpoint has failed");
    if ($CFG->wipCkptName !~ m/^null/i && $CFG->wipCkptName ne '') {
       writeCkpt($CFG->wipCkptName, CKPTFAIL); 
       writeCkpt("ROOTCRS_STACK", CKPTFAIL);
    }
  }
  die @_;
}

sub dieformat
{
  print_error(@_);
  return "Died";
}

# Execute a system command and analyze the return codes
sub system_cmd {
  my $rc = 0;

  trace("Executing @_");
  my @output = system_cmd_capture("@_");

  $rc = shift @output;

  print_lines(@output);

  return $rc;
}

=head2 system_cmd_capture1

  Capture the output from a system command and analyze the return codes

=head3 Parameters

   trace flag to control error output and Command to be executed

=head3 Returns

  Array containing both the return code and the captured output 
  The command output is chomped

=head3 Usage

  To capture the data of command foo:
    my @out = system_cmd_capture1('foo')
    my $rc = shift @out;

  The @out now contains only the output of the command 'foo'

=cut

sub system_cmd_capture1 {
  my $trcflag = shift;

  if ($trcflag) { return system_cmd_capture(@_); }
  else { return system_cmd_capture_noprint(@_); }
}

#Capture the output from a system command
sub system_cmd_capture {
  my @output = system_cmd_capture_noprint(@_);
  my $rc     = shift @output;

  if (scalar(@output) > 0) {
    trace(join("\n>  ", ("Command output:", @output)),
          "\n>End Command output");
  }

  return ($rc, @output)
}



=head2 system_cmd_capture_noprint

  Capture the output from a system command and analyze the return codes

=head3 Parameters

   Command to be executed

=head3 Returns

  Array containing both the return code and the captured output 
  The command output is chomped

=head3 Usage

  To capture the data of command foo:
    my @out = system_cmd_capture('foo')
    my $rc = shift @out;

  The @out now contains only the output of the command 'foo'

=cut

sub system_cmd_capture_noprint {
  my $rc  = 0;
  my $prc = 0;
  my @output;

  trace("Executing cmd: @_");

  if (!open(CMD, "@_ 2>&1 |")) { $rc = -1; }
  else {
    @output = (<CMD>);
    close CMD;
    # the code return must be after the close
    $prc = $CHILD_ERROR >> 8; # get program return code right away
    chomp(@output);
  }

  if ($prc != 0) {
    # program returned error code
    # error("Command return code of $prc from command: @_");
    $rc = $prc;
  }
  elsif ($rc < 0 || ($rc = $CHILD_ERROR) < 0)  {
    my $mycmd = join(' ', @_);
    print_error(180, $mycmd, $rc);
    # It is almost certainly wrong to keep this output from going
    # directly to the trace file, which will occur if not careful,
    # because it will be difficult to determine the cause of serious
    # problems, but this is by request
    push @output,
    "Failure in execution (rc=$rc, $CHILD_ERROR, $!) for command @_";
  }
  elsif ($rc & 127) {
    # program returned error code
    my $sig = $rc & 127;
    my $mycmd = join(' ', @_);
    print_error(187, $sig, $mycmd);
    # It is almost certainly wrong to keep this output from going
    # directly to the trace file, which will occur if not careful,
    # because it will be difficult to determine the cause of serious
    # problems, but this is by request
    push @output,
    "Failure with signal $sig from command: @_";
  }
  elsif ($rc)  { trace("Failure with return code $rc from command @_"); }

  return ($rc, @output);
}

sub print_error
####---------------------------------------------------------
#### Function for dumping NLS error messages on STDOUT
#
# ARGS : 1..n
# ARG1 : NLS message ID 
# ARG2..n : Argments put in NLS message
# Returns : SUCCESS or FAILED 
# Example : print_error(1000, "$orahome", "ocrlocation");
{
  my $errCode = $_[0];
  my @msgArgs = @_[1..$#_];
  my $rc      = -1;
  my @out     = ();
  my @cmd     = ();
  my $ret     = SUCCESS;
  my $nlsArgs;
  my ($sec, $min, $hour, $day, $month, $year) =  (localtime) [0, 1, 2, 3, 4, 5];
  $month = $month + 1;
  $year  = $year  + 1900;
  my $platform = s_get_platform_family();
  my $CLSECHO;

  if ($platform eq 'windows') {
     $CLSECHO = crs_exec_path("clsecho.exe");
  }
  else {
     $CLSECHO = crs_exec_path("clsecho");
  }

  if (! (-x $CLSECHO))
  {
    trace("The CRS executable file 'clsecho' does not exist.");
    return FAILED;
  }

  $ENV{'NLS_LANG'} = $CFG->params('LANGUAGE_ID');

  if (scalar(@msgArgs) > 0)
  {
    $nlsArgs = join('" "', @msgArgs);
    @cmd = ($CLSECHO, "-p has", "-f clsrsc", "-m $errCode", "\"$nlsArgs\"");
  }
  else
  {
     @cmd = ($CLSECHO, "-p has", "-f clsrsc", "-m $errCode");
  }
  @out = system_cmd_capture(@cmd);
  $rc = shift @out;

  if (0 == $rc) {
    # Print out the error message to the console
    printf "$year/%02d/%02d %02d:%02d:%02d @out\n",
      $month, $day, $hour, $min, $sec;

  }
  else {
    trace("Failed to execute the command \"@cmd\" (rc=$rc), ".
           "with the message:\n".join("\n", @out)."\n");
    $ret = FAILED;
  }

  # Dump errors to the log file
  if ($CFG && $CFG->crscfg_trace && $CFG->crscfg_trace_file) 
  {
    undef $NLS_LANG;

    if (scalar(@msgArgs) > 0)
    {
      @cmd = ($CLSECHO, "-p has", "-f clsrsc", "-m $errCode", "\"$nlsArgs\"");
    }
    else
    {
      @cmd = ($CLSECHO, "-p has", "-f clsrsc", "-m $errCode");
    }
    @out = system_cmd_capture(@cmd);
    $rc = shift @out;

    if (0 == $rc) {
      if(!$CFG->AUTO)
      {
        print color 'reset';
        # When printing to a file, the 'print color' prints an unprintable
        # character that will show up as the first character of the next
        # line, making it less readable.  Add newline to make next line
        # more readable
        print "\n";
      }
      trace(@out);
    }
    else {
      trace("Failed to execute the command \"@cmd\" (rc=$rc), ".
            "with the message:\n".join("\n", @out)."\n");
    }     

    $ENV{'NLS_LANG'} = $CFG->params('LANGUAGE_ID');
  }

  if ($CFG->DEBUG) {
    trace("###### Begin Error Stack Trace ######");
    backtrace();
    trace("####### End Error Stack Trace #######\n");
  }

  return $ret;
}

sub print_config
{
  my $cfg = shift;
  my @cfgfiles = ($cfg->paramfile);

  if ($cfg->osdfile && -e $cfg->osdfile) { push @cfgfiles, $cfg->osdfile; }
  if ($cfg->addfile && -e $cfg->addfile) { push @cfgfiles, $cfg->addfile; }

  trace ("### Printing the configuration values from files:");
  for my $file (@cfgfiles) { trace("   $file"); }

  # validates if any value is assigned to the script variables
  for my $key (sort(keys %{$cfg->params})) {
    my $val = $cfg->params($key);
    trace("$key=$val");
  }

  trace ("### Printing other configuration values ###");
  my %cfgh = %{($cfg)};
  for my $key (sort(keys %cfgh)) {
    my $ref = ref($cfg->$key);
    my $val = $cfgh{$key};

    if (!$ref) { trace("$key=$val"); } # scalar
    elsif ($ref eq "ARRAY") { trace("$key=" . join(' ', @{($val)})); }
    elsif ($ref eq "HASH" && $key ne "params" &&
           scalar(keys(%{($val)}))) {
      trace("Printing values from hash $key");
      my %subh = %{($val)};
      for my $hkey (sort(keys(%subh))) {
        trace("  $key key $hkey=$subh{$hkey}");
      }
    }
  }

  trace ("### Printing of configuration values complete ###");

  return;
}

# Define our version of system to ensure proper diagnostics are obtained
# always
sub system {
  return system_cmd(@_);
}

sub read_file
{
    my $file = $_[0];
    open (FILE, "<$file") or die(dieformat(185, $file, $!));
    my @contents = (<FILE>);
    close (FILE);

    return @contents;
}

sub run_crs_cmd {
  my $exec = shift;
  my @cmd = (crs_exec_path($exec), @_);

  return system_cmd(@cmd);
}

sub crs_exec_path {
  my ($cfg, $name, $execpath, $home);

  $cfg = $name = shift;
  if (blessed($cfg))
  {
    # called as a method
    $name = shift;
  }
  else
  {
    $cfg = $CFG;
  }
  $home = shift;

  if ((defined $CFG) && ($CFG->ORA_CRS_HOME))
  {
    if($home ne "")
    {
      $execpath = catfile($home, 'bin', $name);
    }
    else
    {
      $execpath = catfile($cfg->ORA_CRS_HOME, 'bin', $name);
    }

    if ($CFG->platform_family eq 'windows') 
    {
      if (!(-x $execpath) && !(-x "${execpath}.exe") && !(-x "${execpath}.bat"))
      {
        trace (" The CRS executable file $execpath either does not exist or is not executable"); 
      }
    }

    else 
    {
      if (!(-x $execpath))
      {
        trace (" The CRS executable file $execpath either does not exist or is not executable"); 
      }
    }
  }

  return $execpath;
}

# If an element is defined as a counter, return incremented value
# Basically, this is the equivalent of ++counter
sub increment_counter {
  my $cfg  = shift;
  my $name = shift;
  my $incr = 1;
  my $ret;
  if (@_) { 
    $incr = shift;
    if (@_) { croak "Too many args to increment_counter $name"; }
  }
  
  $ret = $cfg->{$name} + $incr;
  $cfg->{$name} = $ret;
 
  return $ret;
}

# ARGS
# ARG 0: paramfile
sub setup_param_vars
{
    my $paramfile = $_[0];
    my $platform  = s_get_platform_family();

    # To support the use of 'strict', it is necessary to create a small
    # program to be executed via 'eval'
    # Because 'strict' requires all variables to be declared, the scope
    # of 'my' variables is the program, and variables that are defined
    # by the parameter file are used in the definition of subsequent
    # parameter file entries, e.g. DIRPREFIX, to get the scoping right
    # it is necessary to create a program that will allow previously
    # defined values
    my @epgm;
    open(PARAMF, $paramfile) or die(dieformat(185, $paramfile, $!));

    while (<PARAMF>) {
        if ($_ !~ /^#|^\s*$/) {
            # The magic below takes params of the form KEY=VAL and sets them as
            # variables in the perl context
            chomp;
            $_ = trim ($_);
            my ($key, $val) = split ('=');

            # store this in a hash that is returned
            if ((0 > index($val,'"')) && $key ne 'ASM_DISK_GROUP') {
              # escape \ (for NT)
              $val =~ s!\\!\\\\!g;
              push @epgm, "my \$$key=\"$val\";";
            } else { # won't allow perl var subst
              # escape "'"
              $val =~ s!\'!\\'!g;
              push @epgm, "my \$$key='$val';";
            }
            push @epgm, '$CFG->params(', "'$key',\$$key);";
        }
    }

    close (PARAMF);
    eval("@epgm");

    # if there was an error log it
    if ($@) { trace($@); }

    return;
}

####---------------------------------------------------------
#### Validating OCR locations based on existing ocr settings
# ARGS: 3
# ARG1 : Path for Oracle CRS home
# ARG2 : Cluster name
# ARG3 : Comma separated OCR locations
sub validateOCR
{
   my $crshome      = $_[0];
   my $clustername  = $_[1];
   my $ocrlocations = $_[2];
   my $status       = SUCCESS;

   if (!$crshome) {
      print_error(4);
      return FAILED;
   }

   if (!(-d $crshome)) {
      print_error(22, $crshome);
      return FAILED;
   }

   trace ("Oracle CRS home = $crshome");

   if (!($clustername)) {
      print_error(6);
      return FAILED;
   }

   trace ("Oracle cluster name = $clustername");
   if (isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST'))) {
      if (! s_copyOCRLoc()) {
         print_error(101);
         return FAILED;
      }
   }
   elsif (! $ocrlocations) {
      print_error(8);
      return FAILED;
   }

   trace ("OCR locations = $ocrlocations");
   trace ("Validating OCR");
   return s_validateOCR ($crshome, $clustername, $ocrlocations);
}

sub validateVarList
####---------------------------------------------------------
#### Function for validating installer variables list
# This validates parameters by checking to see that all have had proper
# substituion.
#
# ARGS: 1
# ARG1: The list of required parameters that should be verified.
# If no argument is specified, this funtion validates all variables
# in parameter file
{
   my (@required) = @_;
   my %params     = %{($CFG->params)};
   my $all_set    = SUCCESS;
   my @keys       = ();

   if (@required) {
      @keys = @required; 
   }
   else {
      @keys = keys(%params);
   }

   trace("Checking parameters from paramfile " . $CFG->paramfile .
         " to validate installer variables");

   # validates if any value is assigned to the script variables
   for my $key (sort(@keys)) {
      my $val = $CFG->params($key);
      {
        if(((@required) && (!$val)) || ($val =~ /^%/)) 
        {
          # The executable 'clsecho' is not yet available at this point
          # hence the print_error(...) doesn't work
          error("No value set for the parameter $key. ",
                "Use parameter file ", $CFG->paramfile, " to set values");
          $all_set = FAILED;
        }
      }
   }

   return $all_set;
}

sub isAddNode
#-------------------------------------------------------------------------------
# Function: Check if hostname is a new node.
# Args    : [0] Name of host to check
#           [1] List of nodes in config
# Returns : TRUE  if     new node
#           FALSE if not new node
#-------------------------------------------------------------------------------
{
   my $hostname = $_[0];
   my $nodelist = $_[1];

   if ($CFG->defined_param('CRS_ADDNODE') &&
       $CFG->params('CRS_ADDNODE') eq "true")
   {
      if ($CFG->DEBUG) { trace("CRS_ADDNODE is defined to 'true'"); }

      return TRUE;
   }
   elsif ($nodelist !~ /\b$hostname\b/i) {
      if ($CFG->DEBUG) {
         trace("'$hostname' is not part of existing nodelist:$nodelist");
      }

      return TRUE;
   }

   if ($CFG->DEBUG) { trace("Not an add node scenario"); }

   return FALSE;
}

sub isCkptPropertyExists
{
   my $ckptName      = $_[0];
   my $ckptPropName  = $_[1];

   my $crshome  = $CFG->ORA_CRS_HOME;
   my @capout = ();
   my $rc;
   my $user = $CFG->params('ORACLE_OWNER');
   my $ORACLE_BASE = $CFG->params('ORACLE_BASE');
   my $CKPTTOOL = catfile( $crshome, 'bin', 'cluutil');
   my @program = ($CKPTTOOL, '-ckpt', '-oraclebase', $ORACLE_BASE, '-chkckpt', '-name', $ckptName, '-pname', $ckptPropName);

   my $env = $ENV{'SRVM_TRACE'};
   undef $SRVM_TRACE;

   # run as specific user, if requested
   $rc = run_as_user2($user, \@capout, @program);
   $ENV{'SRVM_TRACE'} = $env;

   if (scalar(grep(/TRUE/, @capout)) > 0)
   {
     trace("The property $ckptPropName for checkpoint:$ckptName is there");
     return SUCCESS;
   }

   if ((0 != $rc) || (scalar(grep(/FALSE/, @capout))) > 0)
   {
     trace("The property $ckptPropName for checkpoint $ckptName does not exist");
     return FAILED;
   }
}

=head2 isACFSSupported

   Determines if this platform is an ACFS supported platform
   by calling 'acfsdriverstate supported'.

=head3 Parameters

   None

=head3 Returns

  TRUE  - ACFS Supported
  FALSE - ACFS Not Supported

=head3 Notes


=cut
sub isACFSSupported
{
   my $ACFS_supported = FALSE;
   my $acfsdriverstate;

   if ($CFG->platform_family eq 'windows') {
      $acfsdriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'acfsdriverstate.bat');
   }
   else {
      $acfsdriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'acfsdriverstate');
   }

   # check if acfs is supported
   if (! (-e $acfsdriverstate)) {
      trace ("$acfsdriverstate not found");
      return FALSE;
   }

   my @cmd = ($acfsdriverstate, "supported");
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;

   if ($rc == 0) {
      $ACFS_supported = TRUE;
      trace ("acfs is supported");
   }
   else {
      $ACFS_supported = FALSE;
      trace ("acfs is not supported");
   }

  return $ACFS_supported;
}

sub getCkptPropertyValue
{
   my $ckptName     = $_[0];
   my $ckptPropName = $_[1];
   my ($pname, $pval);
   my @capout       = ();
   my @out          = ();
   my $CKPTTOOL     = catfile($CFG->ORA_CRS_HOME, 'bin', 'cluutil');
   my @program      = ($CKPTTOOL, '-ckpt', '-oraclebase',
                       $CFG->params('ORACLE_BASE'), '-listckpt',
                       '-name', $ckptName, '-pname', $ckptPropName);

   my $env = $ENV{'SRVM_TRACE'};
   undef $SRVM_TRACE;

   # run as specific user, if requested
   my $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@capout, @program);
   $ENV{'SRVM_TRACE'} = $env;
   if (0 != $rc) {
      trace("Failed to get property value for property:$ckptPropName " .
            "for checkpoint:$ckptName. Error code is $rc");
      print_error(178, $ckptPropName, $ckptName, $rc);
      return FAILED;
   }
   else {
      trace("Succeeded to get property value:@capout");
      @out = grep(/=/, @capout);
      ($pname, $pval) = split(/=/,$out[0]); 
   }

   return $pval;
}

sub isOwnerGroupValid
#-------------------------------------------------------------------------------
# Function: Validate owner and group
# Args    : none
# Returns : TRUE - if owner and group is     valid
#           FALSE- if owner and group is NOT valid
#-------------------------------------------------------------------------------
{
   # validate owner
   my $owner = $CFG->params('ORACLE_OWNER');
   if (($CFG->FORCE) && ($owner =~ "%")) {
      return FALSE;
   }

   # validate group
   my $group = $CFG->params('ORA_DBA_GROUP');
   if (($CFG->FORCE) && ($group =~ "%")) {
      return FALSE;
   }

   return TRUE;
}

sub srvctl_tty
#-------------------------------------------------------------------------------
# Function: Run SRVCTL command with the given arguments in some cases where the
#           SRVCTL command is invoked remotely via SSH and the arguments include
#           a name that needs to be resolved - See bug 14768261 for details.
# Args    : [0] - TRUE  if run as ORACLE_OWNER
#               - FALSE if run as root
#           [1] - srvctl arguments
#           [2] - CRS home to use. Defaults to $CFG->ORA_CRS_HOME.
# Returns : TRUE  if successful
#           FALSE if failed
#-------------------------------------------------------------------------------
{
   my $run_as_owner = $_[0];
   my $srvctl_args  = $_[1];
   my $home         = $_[2];
   my ($srvctlbin, $status);

   if ($home ne "")
   {
      $srvctlbin = crs_exec_path('srvctl', $home);
   }
   else
   {
      $srvctlbin = crs_exec_path('srvctl');
   }

   # set trace file 
   my $srvctl_trc_file = catfile ($CFG->ORA_CRS_HOME, 'cfgtoollogs', 'crsconfig',
                                  "srvmcfg" . $CFG->srvctl_trc_suff_pp . ".log");

   $ENV{SRVM_TRACE}       = "TRUE";
   $ENV{SRVCTL_TRACEFILE} = $srvctl_trc_file;

   my $cmd = "echo ${srvctlbin} $srvctl_args | bash /dev/stdin";
   trace ("Invoking \"${cmd}\"");
   trace ("trace file=$srvctl_trc_file");

   if ($run_as_owner) {
      $status = run_as_user ($CFG->params('ORACLE_OWNER'), ${cmd});
   }
   else {
      $status = system_cmd(${cmd});
   }

   # set owner & permission of trace file
   s_set_ownergroup ($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'),
                     $srvctl_trc_file);
   s_set_perms ("0775", $srvctl_trc_file);

   #Bug 9718507 . srvctl return code 2 is now only for warnings that can
   #be ignored
   if ((0 == $status) || (2 == $status)) {
      return TRUE;
   } else {
      trace ("  \"${cmd}\" failed with status ${status}.");
      return FALSE;
   }
}

sub srvctl
#-------------------------------------------------------------------------------
# Function: Run srvctl with the given arguments.
# Args    : [0] - TRUE  if run as ORACLE_OWNER
#               - FALSE if run as root
#           [1] - srvctl arguments
#           [2] - CRS home to use. Defaults to $CFG->ORA_CRS_HOME.
# Returns : TRUE  if successful
#           FALSE if failed
#-------------------------------------------------------------------------------
{
   my $run_as_owner = $_[0];
   my $srvctl_args  = $_[1];
   my $home         = $_[2];
   my ($srvctlbin, $status);

   if ($home ne "")
   {
      $srvctlbin = crs_exec_path('srvctl', $home);
   }
   else
   {
      $srvctlbin = crs_exec_path('srvctl');
   }
 
   # set trace file 
   my $srvctl_trc_file = catfile ($CFG->ORA_CRS_HOME, 'cfgtoollogs', 'crsconfig',
                                  "srvmcfg" . $CFG->srvctl_trc_suff_pp . ".log");

   $ENV{SRVM_TRACE}       = "TRUE";
   $ENV{SRVCTL_TRACEFILE} = $srvctl_trc_file;

   my $cmd = "${srvctlbin} $srvctl_args";
   trace ("Invoking \"${cmd}\"");
   trace ("trace file=$srvctl_trc_file");

   if ($run_as_owner) {
      $status = run_as_user ($CFG->params('ORACLE_OWNER'), ${cmd});
   }
   else {
      $status = system_cmd(${cmd});
   }

   # set owner & permission of trace file
   s_set_ownergroup ($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'),
                     $srvctl_trc_file);
   s_set_perms ("0775", $srvctl_trc_file);

   #Bug 9718507 . srvctl return code 2 is now only for warnings that can
   #be ignored
   if ((0 == $status) || (2 == $status)) {
      return TRUE;
   } else {
      trace ("  \"${cmd}\" failed with status ${status}.");
      return FALSE;
   }
}

# Module for script instantiation
#
# Instantiate all files in $CH/crs/sbs/ directory -- replace %FOO%
# with value for FOO (obtained from crsconfig_params) -- and place
# this in $CH/crs/utl/ directory
#
# ARGS
# none
sub instantiate_scripts
{
    my $home = $_[0];
    my $myHome = ($home) ? $home : ($CFG->ORA_CRS_HOME);

    trace("Instantiating scripts in GI home: $myHome");

    my $sbsdir      = catfile ($myHome, "crs", "sbs");
    my @sbsfiles    = glob (catfile ($sbsdir, "*.sbs"));
    my $wrapdir_crs = catfile ($myHome, "crs", "utl");

    # create $wrapdir_crs if it doesn't exist already
    create_dir ($wrapdir_crs);

    foreach my $srcfile (@sbsfiles) {
        my @sbsfile = read_file ($srcfile);

        # strip off .sbs suffix
        (my $dstfile = basename ($srcfile)) =~ s/\.sbs//g;
        my $dstpath = catfile ($wrapdir_crs, $dstfile);
        if ($CFG->DEBUG) { trace ("SRC FILE: $srcfile; DST PATH: $dstpath"); }

        open (DSTPATH, ">${dstpath}")
            or die(dieformat(207, $dstpath, $!));

        foreach my $line (@sbsfile) {
            # skip blanks and comments
            if ($line !~ /^#|^\s*$/) {
                instantiate_config_params ($line);
            }
            print DSTPATH "$line";
        }

        close (DSTPATH);
    }
}

#### Function for creating a directory
# ARGS:
# arg 0 -- directory path to be created
#
# Returns: list of directories -- including intermediate directories -- created
sub create_dir
{
    my $dir_path = $_[0];

    # convert '\' to '/' (for NT)
    $dir_path =~ s!\\!/!g;

    # If dir_path doesn't already exist, create it.
    #
    # If dir_path exists as a symlink, then if the target of the symlink
    # doesn't exist, create the target path.  This is applicable especially
    # to ADE environments where we might already have a symlink pointing to
    # some directory in the has_work/ tree.
    my $link_path;
    if ($link_path = s_isLink ($dir_path)) {
        if ($CFG->DEBUG) {
            trace ("  $dir_path is a SYMLINK to $link_path; changing cwd to " .
                   dirname ($dir_path) . " and resetting DIR_PATH");
        }
        # In ADE env, the target could contain a 'virtual' directory
        # marked by the two dots
        chdir(dirname($dir_path));
        $dir_path = $link_path;
    }

    my @dirs;

    if (!(-e $dir_path)) {
        if ($CFG->DEBUG) { trace ("  mkpath ($dir_path)");}
        @dirs = mkpath ($dir_path)
                  or die(dieformat(254, $dir_path, $!));
    }

    return @dirs;
}

sub instantiate_config_params
{
    # If it contains a pattern of the form '%foo%' AND a mapping exists
    # for 'foo', replace '%foo%' with the corresponding value.
    my $rexp="[a-zA-Z_]+";
    foreach (@_) {
      my @matchlist = $_ =~ /%(${rexp})%/g;
      foreach my $match (@matchlist) {
        if (defined($CFG->config_value($match))) {
          my $sub = $CFG->config_value($match);
          $_ =~ s/%(${match})%/$sub/g;
        }
        elsif ($CFG->defined_param($match)) {
          my $sub = $CFG->params($match);
          $_ =~ s/%(${match})%/$sub/g;
        }
      }

      # We strict the $* instantiation only for the dev env.
      next if (! is_dev_env());
 
      @matchlist = $_ =~ /\$(${rexp})/g;
      foreach my $match (@matchlist) {
        if ($CFG->config_value($match)) {
          my $sub = $CFG->config_value($match);
          $_ =~ s/\$(${match})/$sub/g;
        }
        elsif ($CFG->defined_param($match)) {
          my $sub = $CFG->params($match);
          $_ =~ s/\$(${match})/$sub/g;
        }
      }
    }
}

sub add_ITDIR_to_dirs
#---------------------------------------------------------------------
# Function: add IT_DIR directory to crsconfig_dirs
# Args    : none
#---------------------------------------------------------------------
{

   if ($CFG->defined_param('IT_DIR')) {
      if (is_dev_env()) {
         my $itdir     = $CFG->params('IT_DIR');
         my $SUPERUSER = $CFG->SUPERUSER;
         my $dirsfile  = catfile($CFG->ORA_CRS_HOME, 'crs', 'utl', 'crsconfig_dirs');
         my $platform  = s_get_platform_family();

         open (DIRSFILE, ">>$dirsfile")
              or die(dieformat(208, $dirsfile, $!));
         print DIRSFILE "$platform $itdir $SUPERUSER $SUPERUSER 0755\n";
         close (DIRSFILE);
      }
   }
}

sub create_dirs
{
    # Directories Creation module
    #
    # Create directories with ownership/permissions as specified in
    # crs/utl/crsconfig_dirs
    my $dirs = shift;

    foreach my $line (@$dirs) {
        chomp ($line);
        next if ($line =~ /^#|\$|^\s*$/);  # skip blanks, comments, and lines that 
                                           # unable to instantiate

        if (($CFG->SIHA) && (($line =~ /network\/admin/) || ($line =~ /diag\/tnslsnr/)))
        {
          trace("LINE is $line");
          trace("Skip listener related dirs for SIHA env");
          next;
        }

        # replace variables in input line
        my @matches = $line =~ /(\$\w+)/g;
        for my $match (@matches) {
          if (defined($CFG->config_value($match))) {
            my $sub = $CFG->config_value($match);
            $line =~ s/${match}/$sub/g;
          } elsif ($CFG->defined_param($match)) {
            my $sub = $CFG->params($match);
            $line =~ s/${match}/$sub/g;
          }
        }

        if ($CFG->DEBUG) { trace ("crsconfig_dirs: LINE is $line"); }

        my ($platform, $dir_path, $owner, $grp, $perms) = split (/ /, $line);
        my $myplatformfamily = s_get_platform_family ();
        $myplatformfamily =~ tr/A-Z/a-z/;

        if (($platform eq "all") || ($platform =~ m/$myplatformfamily/)) {

            my @dirs_created = create_dir ($dir_path);

            # if no dir was created, add dir_path to list to set ownership/perms
            # below
            if (!@dirs_created) {
                if ($CFG->DEBUG) {
                    trace ("  no dir created; adding $dir_path to list");
                }
                push (@dirs_created, $dir_path);
            }

            # Setting same ownership/permissions for all intermediate dirs
            # as well
            if (@dirs_created) {
                foreach my $dir (@dirs_created) {
                    # Skip setting ownership and permissions of the /etc/init.d directory on AIX
                    # because this directory doesn't exist on AIX 
                    # On NT, s_set_ownergroup and s_set_perms have no op
                    if ($CFG->platform_family ne "windows") {
                       next if (($OSNAME eq 'aix') && (trim($dir) eq trim($CFG->params('ID')))); 
                       # Change permissions on /etc/init.d only for the test environment
                       next if ((! is_dev_env()) && (trim($dir) eq trim($CFG->params('ID')))); 

                       s_set_ownergroup ($owner, $grp, $dir)
                          or die(dieformat(152, $dir));
                       if ($perms) {
                          s_set_perms ($perms, $dir) or die(dieformat(153, $dir));
                       }
                    }
                }
            }

        }
    }
}

sub copy_wrapper_scripts
{
    #
    # Wrapper copy module
    #
    # Copy files from SOURCE to DEST as specified in
    # crs/utl/crsconfig_files
    #
    my @wcfile = read_file(catfile($CFG->ORA_CRS_HOME,
                                   'crs', 'utl', 'crsconfig_files'));
    foreach my $line (@wcfile) {
        chomp ($line);
        next if ($line =~ /^#|^\s*$/);  # skip blanks and comments
        my ($platform, $src, $dst) = split (/ /, $line);

        my $myplatformfamily = s_get_platform_family ();
        $myplatformfamily =~ tr/A-Z/a-z/;

        if (($platform eq "all") || ($platform =~ m/$myplatformfamily/)) {
            # If the dest file already exists, first remove it
            if (-e $dst) {
                if ($CFG->DEBUG) { trace ("unlink ($dst)"); }

                unlink ($dst) or print_error(168, $dst, $!);
            }
            if ($CFG->DEBUG) { trace ("copy ($src, $dst)"); }

            copy($src, $dst) or print_error(105, $src, $dst, $!);
        }
    }
}

#
# File permissions module
#
# Set ownership/permissions as specified in
# crs/utl/crsconfig_fileperms (after touching the file, if
# required)
#
# ARGS
# arg 0 -- param hash
sub set_file_perms
{
    my $fpfile     = shift;
    my $myplatform = $CFG->platform_family;
    my ($file_name, $bin_file);

    foreach my $line (@$fpfile) {
        chomp ($line);
        next if ($line =~ /^#|^\s*$/);  # skip blanks and comments

        if (($CFG->SIHA) && ($line =~ /network\/admin/))
        {
          trace("LINE is $line");
          trace("Skip listener related files for SIHA env");
          next;
        }

        # replace variables in input line
        $line =~ s/(\$\w+)/$1/eeg;
        if ($CFG->DEBUG) { trace ("crsconfig_fileperms: LINE is $line"); }

        my ($platform, $file_path, $owner, $grp, $perms) = split (/\s+/, $line);
        if (($platform eq "all") || ($platform =~ m/$myplatform/)) {
            if ( !(-e $file_path) && ( $file_path !~ m/.*acfs.*/ ) 
                                  && ( $file_path !~ m/.*oka.*/  )
                                  && ( $file_path  !~ m/.*afd.*/ ) ) {
                if ($CFG->DEBUG) { trace ("Creating empty file $file_path");}
                # create an empty file
                open (FILEHDL, ">$file_path") 
                   or die(dieformat(255, $file_path, $!));
                close (FILEHDL);
            } elsif (!(-e $file_path) &&  ( $file_path =~ m/.*acfs.*/ )) {
                trace ("skipping  set file perm for acfs file $file_path. File does not exist");
                next;
            } elsif (!(-e $file_path) &&  ( $file_path =~ m/.*oka.*/ )) {
                trace ("skipping  set file perm for oka file $file_path. File does not exist");
                next;
            } elsif (!(-e $file_path) &&  ( $file_path =~ m/.*afd.*/ )) {
                trace ("skipping  set file perm for afd file $file_path. File does not exist");
                next;
            }


            # Set ownership/group
            if ($CFG->DEBUG) {
              trace ("s_set_ownergroup ($owner, $grp, $file_path)");
            }

            $file_name = basename($file_path);

            # in dev env, perms should not be set on symlinks
            my $link_path;
            if ($link_path = s_isLink ($file_path)) {
              $bin_file = TRUE;
            }
            elsif (($file_name =~ m/\.bin/) && ($owner eq $CFG->HAS_USER)) {
               $bin_file = TRUE;
            }
            else {
               $bin_file = FALSE;
            }

            if ($bin_file && is_dev_env()) {
               trace("Development env... Not setting permissions on $file_name");
            }
            else {
               # The bug fix for BUG 12752359 - GRID INSTALL IN SC ZONE FAILED, 
               # ROOT.SH CAN'T CHANGE PERMISSION OF LIBSKGXN2.SO
               if (($file_name =~ /skgxn2/) && ($link_path))
               {
                 trace("Skip changing the ownership and ".
                       "permissions of $file_name");
                 next; 
               }

               # Set ownership/group
               s_set_ownergroup ($owner, $grp, $file_path)
                    or die(dieformat(152, $file_path));

               # Set permissions, if specified
               if ($perms) {
                  if ($CFG->DEBUG) { trace ("s_set_perms ($perms, $file_path)");}
                  s_set_perms ($perms, $file_path)
                        or die(dieformat(153, $file_path));
               }
            }
        }
    }

   if (! is_dev_env() && (! $CFG->SIHA)) {
      if ($myplatform eq "unix") {
         if ((!$CFG->ASM_STORAGE_USED) && (! isOCRonASM())) {
            # in an upgrade, OCR_LOCATIONS would be empty.
            if ($CFG->UPGRADE) {
               mod_perms_ocr_loc();
            }
            else {
               my @ocr_locs = split (/\s*,\s*/, $CFG->params('OCR_LOCATIONS'));
               foreach my $loc (@ocr_locs) {
               # Set owner/group of OCR path to root/dba
               trace ("set owner/group of OCR path");
                  if (-e $loc) {
                     s_setParentDirOwner ($CFG->SUPERUSER, $loc);
                  }
               }
            }
         }
      }
   }
}

sub mod_perms_ocr_loc
{
   my $ocrconfig_loc = s_get_config_key('ocr', 'ocrconfig_loc');
   my $ocrmirror_loc = s_get_config_key('ocr', 'ocrmirrorconfig_loc');
   trace ("ocrconfig_loc=$ocrconfig_loc");
   trace ("ocrmirror_loc=$ocrmirror_loc");

   if ($ocrconfig_loc) {
      # check if it's a symbolic link
      if (-l $ocrconfig_loc) {
         my $abs_ocr = s_getAbsLink($ocrconfig_loc);
         s_setParentDirOwner ($CFG->SUPERUSER, $abs_ocr);
      }
      else {
         if (-e $ocrconfig_loc) {
            s_setParentDirOwner ($CFG->SUPERUSER, $ocrconfig_loc);
         }
      }
   }

   if ($ocrmirror_loc) {
      if (-l $ocrmirror_loc) {
         my $abs_ocrmirror = s_getAbsLink($ocrmirror_loc);
         s_setParentDirOwner ($CFG->SUPERUSER, $abs_ocrmirror);
      }
      else {
         if (-e $ocrmirror_loc) {
            s_setParentDirOwner ($CFG->SUPERUSER, $ocrmirror_loc);
         }
      }
   }
}

####---------------------------------------------------------
#### Function to check if OCR is on ASM
sub isPathonASM
{
   trace ("Checking if given path is on ASM");

   my $diskpath = $_[0];

   if (!$diskpath) {
      trace ("Device path is not specified");
      return FALSE;
   }

   if ($diskpath =~ /\+/) {
      return TRUE;
   } 
   else {
      return FALSE;
   }
}

sub isRAC_appropriate
#-------------------------------------------------------------------------------
# Function:  Check if rac_on/rac_off on Unix
# Args    :  none
# Returns :  TRUE  if rac_on/rac_off     needs to be set
#            FALSE if rac_on/rac_off not needs to be set
#-------------------------------------------------------------------------------
{
   my $myplatformfamily = s_get_platform_family ();
   $myplatformfamily =~ tr/A-Z/a-z/;

   if ($myplatformfamily eq "unix") {
      return s_isRAC_appropriate ();
   } 
   else {
      return TRUE
   }
}

sub set_perms_ocr_vdisk
{
   my $platform = $CFG->platform_family;
   my ($line, @dirs, @files);

   # add OCR_LOCATION and OCR_MIRROR_LOCATION
   if (! $CFG->SIHA) {
      # OCR permissions need to change to 0600 when CSSD dependency on OCR
      # goes away. Bypass if ASM is used.

      if (!$CFG->ASM_STORAGE_USED) {
         my @ocr_locs = split (/\s*,\s*/, $CFG->params('OCR_LOCATIONS'));
         my $loc;
         foreach $loc (@ocr_locs) {
            $line = join (' ', $platform, $loc,
                          $CFG->SUPERUSER, $CFG->params('ORA_DBA_GROUP'), '0640');
            push @files, $line;

            # set owner and permission of OCR directory
            my @loc_dirs;
            if ($platform eq "windows") {
               @loc_dirs = split (/\\/, $loc);
            }
            else {
               @loc_dirs = split (/\//, $loc);
            }

            my $nbr_of_levels = scalar (@loc_dirs);
            # $nbr_of_levels = 2 means it's at the root directory (exp: R:\ocr).
            # Therefore, no need to add to crsconfig_dirs
            if ($nbr_of_levels > 2 ) {
               my ($dir) = split ($loc_dirs[$nbr_of_levels-1], $loc);
               $line     = join (' ', $platform, $dir, $CFG->params('ORACLE_OWNER'),
                                 $CFG->params('ORA_DBA_GROUP'), '0755');
               push @dirs, $line;
           }
         }
      }
   }

   # add all voting disks.  Bypass if ASM is used.
   # XXX: is this step required? Existing shell scripts don't seem to be
   # using validate_VDisks() function
   if (!$CFG->ASM_STORAGE_USED) {
      my @votingdisks = split (/\s*,\s*/, $CFG->params('VOTING_DISKS'));

      my $vdisk;
      foreach $vdisk (@votingdisks) {
         # voting disk should not be precreated since the startup script may
         # be run as a different user than crs user. Precreating/touching
         # a voting disk prematurely will cause later I/Os to fail, such as
         # voting file upgrade/create
         #
         # set owner and permission of votind disks directory by adding to
         # crsconfig_dirs
         my @vdisk_dirs;
         if ($platform eq "windows") {
            @vdisk_dirs = split (/\\/, $vdisk);
         }
         else {
            # other platforms
            @vdisk_dirs = split (/\//, $vdisk);
         }

         my $nbr_of_levels = scalar (@vdisk_dirs);
         # $nbr_of_levels = 2 means it's at the root directory (exp: R:\vdisk).
         # Therefore, no need to add to crsconfig_dirs
         if ($nbr_of_levels > 2 ) {
            my ($dir) = split ($vdisk_dirs[$nbr_of_levels-1], $vdisk);
            $line     = join (' ', $platform, $dir, $CFG->params('ORACLE_OWNER'),
                              $CFG->params('ORA_DBA_GROUP'), '0755');
            push @dirs, $line;
         }
      }
   }

   # add OCRCONFIGDIR and OLRCONFIGDIR to crsconfig_dirs
   my $owner;
   if (is_dev_env()) {
      $owner = $CFG->params('ORACLE_OWNER');
   } else {
      $owner = $CFG->SUPERUSER;
   }

   if ($CFG->defined_param('OCRCONFIGDIR')) {
      $line = join (' ', $platform, $CFG->params('OCRCONFIGDIR'),
                    $owner, $CFG->params('ORA_DBA_GROUP'), '0755');
      push @dirs, $line;
   }

   if ($CFG->defined_param('OLRCONFIGDIR') && 
       $CFG->defined_param('OCRCONFIGDIR'))
   {
      if ($CFG->params('OLRCONFIGDIR') ne $CFG->params('OCRCONFIGDIR')) {
         $line = join (' ', $platform, $CFG->params('OLRCONFIGDIR'),
                       $owner, $CFG->params('ORA_DBA_GROUP'), '0755');
         push @dirs, $line;
      }
   }

   create_dirs (\@dirs);
   set_file_perms (\@files);
}

sub setperm_vdisks
#-------------------------------------------------------------------------------
# Function: Set voting disks permission
#-------------------------------------------------------------------------------
{
   if (($CFG->platform_family eq 'windows') ||
       ($CFG->ASM_STORAGE_USED) ||
       (isOCRonASM()))
   {
      return SUCCESS;
   }

   my $owner  = $CFG->params('ORACLE_OWNER');
   my $group  = $CFG->params('ORA_DBA_GROUP');
   my $rc     = SUCCESS;
   my @vdisks = extractVotedisks();

   foreach (@vdisks) {
      if ($CFG->DEBUG) { trace ("set permission on $_"); }
      if (! s_set_ownergroup ($owner, $group, $_)) {
         $rc = FAILED;
         print_error(152, $_);
         last;
      }

      if (! s_set_perms("0640", $_)) {
         $rc = FAILED;
         print_error(153, $_);
         last;
      }
   }

   return $rc;
}

sub copydir
{
   my $sourcedir = $_[0];
   my $destdir   = $_[1];
   my ($filepath, $filename, $destfilepath, $dirpath);
   my @filelist;
   trace("source dir=$sourcedir");
   trace("dest   dir=$destdir");

   find(sub { push @filelist, $File::Find::name }, $sourcedir);

   if ($CFG->platform_family eq "windows") {
      # escape char '\' on Windows
      $destdir =~ s!\\!\\\\!g;
      $sourcedir =~ s!\\!\\\\\\\\!g;
   }

   foreach $filepath (@filelist) {
     if ( -f $filepath) {
        $filename = basename($filepath);
        # Do not copy OCR backup files
        if ($filename =~ /\.ocr$/)
        {
          trace("Skipping the file '$filename'");
          next; 
        }

        $dirpath = dirname($filepath);

        if ($CFG->platform_family eq "windows") {
           # escape char '\' on Windows
           $dirpath =~ s!\\!\\\\!g;
        }

        $dirpath =~ s/$sourcedir/$destdir/g;
        if (! -e $dirpath) {
          trace("creating directory $dirpath");
           mkpath( $dirpath);
           s_set_ownergroup ($CFG->params('ORACLE_OWNER'),
                             $CFG->params('ORA_DBA_GROUP'), $dirpath);
        }

        $destfilepath = catfile($dirpath, $filename);
        trace ("copying file=$filename");
        trace ("source file path=$filepath");
        trace ("dest file path=$destfilepath");
        copy_file ($filepath, $destfilepath,
                   $CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'));
     }
   }
}

####---------------------------------------------------------
#### Copy file from one local location to another
#
# Copies file from one local location to another
# if user/group given, will chown copied file as it
# ARGS: 4
# ARG1 : Source file path
# ARG2 : Destination file path
# ARG3 : User owner to set (or undef)
# ARG4 : Group owner to set (or undef)
# @returns SUCCESS or $FAILURE
#
sub copy_file { 
  my $src = $_[0]; 
  my $dst = $_[1]; 
  my $usr = $_[2]; 
  my $grp = $_[3]; 

  if (! (-f $src)) {
     trace("  $src ? -f failed" );
     return FAILED;
  }
  trace("  copy \"$src\" => \"$dst\"" );
  if (! copy( $src, $dst ))
  {
    print_error(105, $src, $dst, $!);
    return FAILED;
  }
  # chown to specific user if requested
  if (defined( $usr ) && defined( $grp )) 
  {
    trace("  set ownership on \"$dst\" => ($usr,$grp)" );
    if (FAILED == s_set_ownergroup ($usr, $grp, $dst)) 
    { 
      print_error(152, $dst);
      return FAILED;
    }
  }
  return SUCCESS;
}

sub getCkptStatus
{
   my $ckptName = $_[0];
   my $crshome  = $CFG->ORA_CRS_HOME;
   my @capout = ();
   my $rc;
   my $user = $CFG->params('ORACLE_OWNER');
   my $ORACLE_BASE = $CFG->params('ORACLE_BASE');
   my $CKPTTOOL = catfile( $crshome, 'bin', 'cluutil');
   my @program = ($CKPTTOOL, '-ckpt', '-oraclebase', $ORACLE_BASE, '-chkckpt', '-name', $ckptName, '-status');

   my $env = $ENV{'SRVM_TRACE'};
   undef $SRVM_TRACE;

   # run as specific user, if requested
   $rc = run_as_user2($user, \@capout, @program);
   $ENV{'SRVM_TRACE'} = $env;
   # cluutil return 0 err code and errors, if any, on stdout
   if (scalar(grep(/START/, @capout)) > 0)
   {
     trace("The '$ckptName' status is START");
     return CKPTSTART;
   }
   elsif (scalar(grep(/SUCCESS/, @capout)) > 0)
   {
     trace("The '$ckptName' status is SUCCESS");
     return CKPTSUC;
   }
   elsif (scalar(grep(/FAIL/, @capout)) > 0)
   {
     trace("The '$ckptName' status is FAILED");
     return CKPTFAIL;
   }
}

sub isCkptSuccess
{
   my $ckptName = $_[0];
   my $crshome  = $CFG->ORA_CRS_HOME;
   my @capout = ();
   my $rc;
   my $user = $CFG->params('ORACLE_OWNER');
   my $ORACLE_BASE = $CFG->params('ORACLE_BASE');
   my $CKPTTOOL = catfile( $crshome, 'bin', 'cluutil');
   my @program = ($CKPTTOOL, '-ckpt', '-oraclebase', $ORACLE_BASE, '-chkckpt', '-name', $ckptName, '-status');

   my $env = $ENV{'SRVM_TRACE'};
   undef $SRVM_TRACE;

   # run as specific user, if requested
   $rc = run_as_user2($user, \@capout, @program);
   $ENV{'SRVM_TRACE'} = $env;
   # cluutil return 0 err code and errors, if any, on stdout
   if (scalar(grep(/SUCCESS/, @capout)) > 0)
   {
     trace("The '$ckptName' status is SUCCESS");
     return TRUE;
   }
   else
   {
     trace("The '$ckptName' is either in START/FAILED state");
     return FALSE;
   }
}

sub isCkptexist
#-------------------------------------------------------------------------------
# Function: Verify if checkpoint exist
# Args    : Check point Name
# Returns : boolean
#-------------------------------------------------------------------------------
{
   my $ckptName = $_[0];
   my $crshome  = $CFG->ORA_CRS_HOME;
   my @capout = ();
   my $rc;
   my $user = $CFG->params('ORACLE_OWNER');
   my $ORACLE_BASE = $CFG->params('ORACLE_BASE');
   my $CKPTTOOL = catfile( $crshome, 'bin', 'cluutil');
   my @program = ($CKPTTOOL, '-ckpt', '-oraclebase', $ORACLE_BASE, '-chkckpt', '-name', $ckptName);
   # run as specific user, if requested
   trace( '     ckpt: '.join(' ', @program) );

   my $env = $ENV{'SRVM_TRACE'};
   undef $SRVM_TRACE;

   $rc = run_as_user2($user, \@capout, @program);
   $ENV{'SRVM_TRACE'} = $env;
   # cluutil return 0 err code and errors, if any, on stdout
   if (scalar(grep(/TRUE/, @capout)) > 0)
   {
     
     return SUCCESS;
   }
   if ((0 != $rc) || (scalar(grep(/FALSE/, @capout))) > 0)
   {
     trace("checkpoint $ckptName does not exist");
     return FAILED;
   }
}


sub writeCkpt
{
   my $ckptName =  $_[0];
   my $ckptState = $_[1];
   my $crshome  = $CFG->ORA_CRS_HOME;
   my @capout = ();
   my $rc;
   my $user = $CFG->params('ORACLE_OWNER');
   my $ORACLE_BASE = $CFG->params('ORACLE_BASE');
   my $CKPTTOOL = catfile( $crshome, 'bin', 'cluutil');
   my @program = ($CKPTTOOL, '-ckpt', '-oraclebase', $ORACLE_BASE, '-writeckpt', '-name', $ckptName, '-state', $ckptState);

   my $env = $ENV{'SRVM_TRACE'};
   undef $SRVM_TRACE;

   # run as specific user, if requested
   $rc = run_as_user2($user, \@capout, @program);
   $ENV{'SRVM_TRACE'} = $env;
   if (0 != $rc)
   {
     trace("Failed to write the checkpoint:'$ckptName' with status:$ckptState.Error code is $rc");
     print_error(175, $ckptName, $ckptState, $rc);
     return FAILED;
   }
   else
   {
     trace("Succeeded in writing the checkpoint:'$ckptName' with status:$ckptState");
     return SUCCESS;
   }
}

sub writeCkptPropertyFile
{
   my $ckptName      = $_[0];
   my $ckptPropFile  = $_[1];

   my $crshome  = $CFG->ORA_CRS_HOME;
   my @capout = ();
   my $rc;
   my $user = $CFG->params('ORACLE_OWNER');
   my $ORACLE_BASE = $CFG->params('ORACLE_BASE');
   my $CKPTTOOL = catfile( $crshome, 'bin', 'cluutil');
   my @program = ($CKPTTOOL, '-ckpt', '-oraclebase', $ORACLE_BASE, '-writeckpt', '-name', $ckptName, '-pfile', $ckptPropFile);

   my $env = $ENV{'SRVM_TRACE'};
   undef $SRVM_TRACE;

   # run as specific user, if requested
   $rc = run_as_user2($user, \@capout, @program);
   $ENV{'SRVM_TRACE'} = $env;
   if (0 != $rc)
   {
     trace("Failed to write contents of pfile :$ckptPropFile for checkpoint:$ckptName. Error code is $rc");
     print_error(176, $ckptPropFile, $ckptName, $rc);
     return FAILED;
   }
   else
   {
     trace("write contents of pfile :$ckptPropFile for checkpoint:$ckptName succeeded");
     return SUCCESS;
   }
}

sub writeCkptProperty
{
   my $ckptName      = $_[0];
   my $ckptPropName  = $_[1];
   my $ckptPropValue = $_[2];
   my $crshome  = $CFG->ORA_CRS_HOME;
   my @capout = ();
   my $rc;
   my $user = $CFG->params('ORACLE_OWNER');
   my $ORACLE_BASE = $CFG->params('ORACLE_BASE');
   my $CKPTTOOL = catfile( $crshome, 'bin', 'cluutil');
   my @program = ($CKPTTOOL, '-ckpt', '-oraclebase', $ORACLE_BASE, '-writeckpt', '-name', $ckptName, '-pname', $ckptPropName, '-pvalue', $ckptPropValue);

   my $env = $ENV{'SRVM_TRACE'};
   undef $SRVM_TRACE;

   # run as specific user, if requested
   $rc = run_as_user2($user, \@capout, @program);
   $ENV{'SRVM_TRACE'} = $env;
   if (0 != $rc)
   {
     trace("Failed to add (property/value):('$ckptPropName/'$ckptPropValue') for checkpoint:$ckptName.Error code is $rc");
     print_error(177, $ckptPropName, $ckptPropValue, $ckptName, $rc);
     return FAILED;
   }
   else
   {
     trace("Succeeded to add (property/value):('$ckptPropName/'$ckptPropValue') for checkpoint:$ckptName");
     return SUCCESS;
   }
}

sub clean_start_cluster
{
  my $ckpt = "ROOTCRS_STRTSTACK";
  trace("Cleaning clusterware stack startup failure");
  my $CRSCTL = crs_exec_path('crsctl');
  my $rc;
  my @out = system_cmd_capture($CRSCTL, "stop", "resource", "-all", "-init");
  $rc = shift @out;
  if ($rc !=0) {
     trace("clean_start_cluster output is @out");
  }
  writeCkpt($ckpt, CKPTSTART);
}

####---------------------------------------------------------
#### Function for 'TOUCH'ing local.olr file if it does not exist
#    It also validates/sets up OLR config if does not exist
# ARGS: 2
# ARG1 : Complete path of OLR location
# ARG2 : CRS Home
sub validate_olrconfig
{
    my $olrlocation = $_[0];
    my $crshome     = $_[1];

    if (!$olrlocation) {
        print_error(9);
        return FAILED;
    }

    if (!(-f $olrlocation)) 
    {
       # create an empty file and reset permission
       if ($CFG->DEBUG) { trace ("create $olrlocation"); }
       open (FILEHDL, ">$olrlocation") or return FAILED;
       close (FILEHDL);

       if ($CFG->SIHA) {
          s_set_ownergroup ($CFG->params('ORACLE_OWNER'),
                            $CFG->params('ORA_DBA_GROUP'), $olrlocation)
               or die(dieformat(152, $olrlocation));
          s_set_perms ("0600", $olrlocation) 
               or die(dieformat(153, $olrlocation));
       } else {
          s_set_ownergroup ($CFG->SUPERUSER, $CFG->params('ORA_DBA_GROUP'),
                            $olrlocation)
               or die(dieformat(152, $olrlocation));
          s_set_perms ("0600", $olrlocation) 
               or die(dieformat(153, $olrlocation));

       }
    }
    trace ("OLR location = " . $olrlocation);

    if (!$crshome) {
        print_error(4);
        return FAILED;
    }

    if (!(-d $crshome)) {
        print_error(5, $crshome);
    }
    trace ("Oracle CRS Home = " . $crshome);

    # OSD to validate OLR config
    my $rc = s_validate_olrconfig ($olrlocation, $crshome);

    return $rc;
}

####---------------------------------------------------------
#### Check if this is a SI CSS configuration
sub validate_SICSS
{
   trace ("Validating for SI-CSS configuration");
   my $ocrfile = get_ocrdisk();

   if (!$ocrfile) {
      trace ("Unable to retrieve ocr disk info");
      return SUCCESS;
   }

   if ($ocrfile =~ /\+/) {
      # return if ocrfile is ASM disk group
      return SUCCESS;
   }
   else {
      if (! (-e $ocrfile)) {
         print_error(13, $ocrfile);
         return SUCCESS;
      }
   }

   # OCR location already specified. Check if it is used for
   # single instance CSS/ASM
   my $local_flag = s_get_config_key("ocr", "local_only");
   if (!$local_flag) {
      return FAILED;
   }

   # convert to upper-case
   $local_flag =~ tr/a-z/A-Z/;

   # Previous installation of 10g single instance
   trace ("LOCAL_FLAG = " . $local_flag);
   if ($local_flag eq "TRUE") {
      print_error(14);
      return FAILED;
   }

   return SUCCESS;
}


=head2 versionComparison

  Make a comparison between two version strings 

=head3 Parameters

  Args: 2
   [0]: version1
   [1]: version2

=head3 Returns

     1: version1 > version2
     0: version1 = version2
    -1: version1 < version2
  
=head3 Usage
  
  my $ret = versionComparison("12.1.0.0.0", "11.1.0.0.0");
    
=cut

sub versionComparison
{
  my $leftVersion  = $_[0];
  my $rightVersion = $_[1];
  my $retVal = 0;

  trace("leftVersion=$leftVersion; rightVersion=$rightVersion");

  if ((!$leftVersion) || (!isValidVersion($leftVersion)))
  {
    die("Missing or invalid version number [$leftVersion]");
  }

  if ((!$rightVersion) || (!isValidVersion($rightVersion)))
  {
    die("Missing or invalid version number [$rightVersion]");
  }

  if ($leftVersion eq $rightVersion)
  {
    trace("[$leftVersion] is same as [$rightVersion]");
    return $retVal;
  }

  my @ver1 = split(/\./, $leftVersion);
  my @ver2 = split(/\./, $rightVersion);

  my $i = 0;
  for (; $i < scalar(@ver1); $i++)
  {
    my $j = 0;
    while (($j < $i) && ($ver1[$j] == $ver2[$j])) { $j++; }

    if (($j == $i) && ($ver1[$j] > $ver2[$j]))
    {
      last;
    }
  }

  if ($i < scalar(@ver1))
  {
    $retVal = 1;
  }
  else
  {
    $retVal = -1;
  }

  my $cmp = ((1 == $retVal) ? 'higher' : 'lower');
  trace("[$leftVersion] is $cmp than [$rightVersion]");

  return $retVal;
}


=head2 isValidVersion 

  Check validity of a version string, which must be in the format A.B.C.D.E  

=head3 Parameters

  Args: 1 
   [0]: Version to be checked

=head3 Returns

   TRUE or FALSE
  
=head3 Usage
  
    
=cut

sub isValidVersion
{
  my $verString = shift;
  my @ver = split(/\./, $verString);
 
  if (scalar(@ver) != 5)
  {
    trace("Not 5-digit version numbers");
    return FALSE;
  }

  foreach my $v (@ver)
  {
    if (($v !~ /^[0-9]+$/) || ($v =~ /^0.+$/))
    {
      trace("Not all version numbers are numeric");
      return FALSE;
    }
  }

  return TRUE;
}

sub isVersion10
{
   my $is10ver;
   my @oldcrs_ver = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   if ($oldcrs_ver[0] eq '10') {
       $is10ver = TRUE;
   } else {
       $is10ver = FALSE;
   }
   return $is10ver;
}

sub isVersion111
{
   my $is111ver;
   my @oldcrs_ver = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   if ($oldcrs_ver[0] eq '11' && $oldcrs_ver[1] eq '1') {
       $is111ver = TRUE;
   } else {
       $is111ver = FALSE;
   }
   return $is111ver;
}

sub isVersion112
{
   my $is112ver;
   my @oldcrs_ver = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   if ($oldcrs_ver[0] eq '11' && $oldcrs_ver[1] eq '2') {
       $is112ver = TRUE;
   } else {
       $is112ver = FALSE;
   }
   return $is112ver;
}

# Function to check if old CRS version is lower than 11.2
sub isOldVersionLT112
{
  my @oldCrsVer = @{$CFG->oldconfig('ORA_CRS_VERSION')};
  my $verinfo = join('.', @oldCrsVer);

  if (-1 == versionComparison($verinfo, "11.2.0.0.0"))
  {
    return TRUE;
  }
  else
  {
    return FALSE;
  }  
}

# Function to check if old CRS version is lower than 11.1
sub isOldVersionLT111
{
  my @oldCrsVer = @{$CFG->oldconfig('ORA_CRS_VERSION')};
  my $verinfo = join('.', @oldCrsVer);

  if (-1 == versionComparison($verinfo, "11.1.0.0.0"))
  {
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}

####---------------------------------------------------------
#### Function to check if OCR is on ASM
sub isOCRonASM
{
    trace ("Checking if OCR is on ASM");

    my $ocrfile   = get_ocrdisk();
    my $ocrmirror = get_ocrmirrordisk();
    my $ocrloc3   = get_ocrloc3disk();
    my $ocrloc4   = get_ocrloc4disk();
    my $ocrloc5   = get_ocrloc5disk();

    if (!$ocrfile) {
        trace ("OCR config does not exist");
        return FALSE;
    }

    if (($ocrfile =~ /\+/) || ($ocrmirror =~ /\+/) || ($ocrloc3 =~ /\+/) ||
        ($ocrloc4 =~ /\+/) || ($ocrloc5 =~ /\+/))
    {
       return TRUE;
    }
    else {
       return FALSE;
    }
}

sub isONSexist
#-------------------------------------------------------------------------------
# Function: Find if ONS is already configured
# Args    : none
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
{
  my $status = SUCCESS;

  trace("Invoking srvctl config ons");
  my $srvctl = crs_exec_path('srvctl');
  my @out    = system_cmd_capture($srvctl, "config", "ons");
  my $rc     = shift @out;

  if ($rc != 0) {
    trace("ONS resource does not exist. output is @out");
    $status = FAILED;
  } else  {
    trace("ONS resource  exist. output is @out");
  }

  return $status;
}

sub get_ocrdisk
#-------------------------------------------------------------------------------
# Function: Retrieving OCR location
# Args    : none
#-------------------------------------------------------------------------------
{
   trace ("Retrieving OCR main disk location");

   if ($CFG->platform_family ne "windows") {
      if (!(-r $CFG->params('OCRCONFIG'))) {
         print_error(103, $CFG->params('OCRCONFIG'));
         return $CFG->params('OCRCONFIG');
      }
   }

   return s_get_config_key("ocr", "ocrconfig_loc");
}

sub get_ocrmirrordisk
#-------------------------------------------------------------------------------
# Function: Retrieving OCR mirror location
# Args    : none
#-------------------------------------------------------------------------------
{
   trace ("Retrieving OCR mirror disk location");

   if ($CFG->platform_family ne "windows") {
      if (!(-r $CFG->params('OCRCONFIG'))) {
         print_error(103, $CFG->params('OCRCONFIG'));
         return $CFG->params('OCRCONFIG');
      }
   }

   return s_get_config_key("ocr", "ocrmirrorconfig_loc");
}

sub get_ocrloc3disk
#-------------------------------------------------------------------------------
# Function: Retrieving OCR location 3
# Args    : none
#-------------------------------------------------------------------------------
{
   trace ("Retrieving OCR loc3 disk location");
   return s_get_config_key("ocr", "ocrconfig_loc3");
}

sub get_ocrloc4disk
#-------------------------------------------------------------------------------
# Function: Retrieving OCR location 4
# Args    : none
#-------------------------------------------------------------------------------
{
   trace ("Retrieving OCR loc4 disk location");
   return s_get_config_key("ocr", "ocrconfig_loc4");
}

sub get_ocrloc5disk
#-------------------------------------------------------------------------------
# Function: Retrieving OCR location 5
# Args    : none
#-------------------------------------------------------------------------------
{
   trace ("Retrieving OCR loc5 disk location");
   return s_get_config_key("ocr", "ocrconfig_loc5");
}

sub get_ocrlocaldisk
#-------------------------------------------------------------------------------
# Function: Retrieving OCR local_only location
# Args    : none
#-------------------------------------------------------------------------------
{
    trace ("Retrieving OCR local_only location");
    return s_get_config_key("ocr", "local_only");
}

sub get_srvdisk
#-------------------------------------------------------------------------------
# Function: Retrieving SRV location
# Args    : none
#-------------------------------------------------------------------------------
{
    trace ("Retrieving SRV location");
    return s_get_config_key("srv", "srvconfig_loc");
}

#------------------------------------------------------------------------------
# Ref: 
# Table 1-1 List of Processes and Services Associated with Oracle Clusterware 
#           Components
#
#------------------------------------------------------------------------------
sub create_pinproc_list
{
   my $pinproc;
   if ($CFG->platform_family eq "unix") 
   {
     if (is_dev_env()) 
     {
       $pinproc = "osysmond,ologgerd,ocssd,cssdmonitor,cssdagent,mdb_pmon_-MGMTDB";
     }
     else
     {
       $pinproc = "osysmond.bin,ologgerd,ocssd.bin,cssdmonitor,cssdagent,mdb_pmon_-MGMTDB";
     }
   }

   if ($^O eq "linux") 
   {
     if (is_dev_env()) 
     {
       $pinproc= "osysmond,ologgerd,ocssd,cssdmonitor,cssdagent,mdb_pmon_-MGMTDB,kswapd0";
     }
     else
     {
       $pinproc= "osysmond.bin,ologgerd,ocssd.bin,cssdmonitor,cssdagent,mdb_pmon_-MGMTDB,kswapd0";
     }
   }

   if ($^O eq "solaris") 
   {
     if (is_dev_env()) 
     {
       $pinproc= "osysmond,ologgerd,ocssd,cssdmonitor,cssdagent,mdb_pmon_-MGMTDB,pageout,sched";
     }
     else
     {
       $pinproc= "osysmond.bin,ologgerd,ocssd.bin,cssdmonitor,cssdagent,mdb_pmon_-MGMTDB,pageout,sched";
     }
   }

   if ($CFG->platform_family eq "windows")
   {
     $pinproc= "osysmond.exe,ologgerd.exe,ocssd.exe,cssdmonitor.exe,cssdagent.exe,mdb_pmon_-MGMTDB";
   }


   trace("Pin process list is $pinproc");
   return $pinproc;
}

sub crf_config_generate
{
  my $mynameEntry = $_[0];
  my $hlist = "";
  my $nodelist = "";
  my $master = "";
  my $replica = "";
  my $masterpub = "";
  my $bdbloc = $_[1]; 
  my $usernm = $_[2];
  my $pinproc= create_pinproc_list(); 
  my $clustnm = $CFG->params('CLUSTER_NAME');
  my $crfhome = $CFG->ORA_CRS_HOME;
  my $host = $CFG->HOST;
  my $configfile;

  (undef, $configfile) = tempfile();

  $hlist=$_[3];
  $hlist =~ s/ //g;
  $nodelist=$_[3];
  chomp($nodelist);
  my @hosts = split(/[,]+/, $nodelist);
  $master = $hosts[0];

  if ($CFG->platform_family eq "windows")
  {
    $usernm = "";
  }
  
  # no replica if less than 2 nodes
  if (scalar(@hosts) >= 2) { $replica = $hosts[1]; }

  if ($mynameEntry eq $master)
  {
    $masterpub = $CFG->HOST;
  }

  my $orafile = catfile ($crfhome, "crf", "admin", "crf$host.ora");

  if (!-e $orafile)
  {
    open CONFIG_FILE,'>',$configfile or die $!;
    print CONFIG_FILE  "BDBLOC=$bdbloc\n" ;
    trace ("Cluster Health Monitor repository location updated with $bdbloc");
    print CONFIG_FILE  "PINNEDPROCS=$pinproc\n" ;
    trace ("Cluster Health Monitor pin process list updated with $pinproc");
  }
  else
  {
    copy_file($orafile, $configfile);
    trace ("Copied existing orafile");
    open CONFIG_FILE,'>>',$configfile or die $!;
    if (getCHMAttrib("BDBLOC", $configfile) eq "")
    {
      print CONFIG_FILE  "BDBLOC=$bdbloc\n" ;
      trace("Cluster Health Monitor repository location copied if not present");
    }
    if (getCHMAttrib("PINNEDPROCS", $configfile) eq "")
    {
      print CONFIG_FILE  "PINNEDPROCS=$pinproc\n" ;
      trace ("Cluster Health Monitor pin process list updated with $pinproc");
    }
  } 

  if (getCHMAttrib("HOSTS", $configfile) eq "")
  {
    print CONFIG_FILE  "HOSTS=$hlist\n" ;
  }
  if (getCHMAttrib("MASTER", $configfile) eq "")
  {
    print CONFIG_FILE  "MASTER=$master\n" ;
  }
  if (getCHMAttrib("MYNAME", $configfile) eq "")
  {
    print CONFIG_FILE  "MYNAME=$mynameEntry\n" ;
  }
  if (getCHMAttrib("MASTERPUB", $configfile) eq "")
  {
    print CONFIG_FILE  "MASTERPUB=$masterpub\n" ;
  }
  if (getCHMAttrib("CLUSTERNAME", $configfile) eq "")
  {
    print CONFIG_FILE  "CLUSTERNAME=$clustnm\n" ;
  }
  if (getCHMAttrib("USERNAME", $configfile) eq "")
  {
    print CONFIG_FILE  "USERNAME=$usernm\n";
  }
  if (getCHMAttrib("CRFHOME", $configfile) eq "")
  {
    print CONFIG_FILE  "CRFHOME=$crfhome\n" ;
  }
  close CONFIG_FILE ;

  return $configfile;
}

sub removeCHMDB
{ 
  my $bdbloc = getCHMAttrib("BDBLOC");                                 
  if ($bdbloc eq "default")
  {
    $bdbloc = catfile($CFG->ORA_CRS_HOME, "crf", "db", $CFG->HOST);
  }
  if (-d $bdbloc)
  {
    # delete old bdb files.                     
    trace("Deleting CHM repository at files at: ", $bdbloc);
    crf_delete_bdb($bdbloc);                                             
  } 
  else                                                          
  {                                                      
    trace("CHM repository path not found");
  }
} 
    
# This subroutine takes argument as attribute of ora file and returns 
# the corresponding value of that attribute in the ora file.
# Eg: If ora file has a line with "BDBSIZE = 25632", 
# this subroutine returns 25632.        
sub getCHMAttrib                                     
{   
  my $arg = $_[0];
  my $orafile = $_[1];
  my $loc;
  my $home = s_get_olr_file ("crs_home");
  my $host = tolower_host();
      
  if ($arg ne "")       
  {                                
    # Read the ora file to get the BDB path                                     
    if ($orafile eq "")
    {
      $orafile = catfile ($home, "crf", "admin", "crf$host.ora");
    }
    if (!-f $orafile)
    {
      trace("Info: No ora file present at ", $orafile);        
      return "";
    }
                                         
    my @filecontent = read_file ($orafile);
    foreach my $line (@filecontent)
    {                                                   
      # skip blanks and comments 
      if ($line !~ /^#|^\s*$/)                                 
      {
        if ($line =~ /$arg=(.*)/) { $loc = $1; last;}
      }
    }
    return $loc;
  }
  return "";
}                                                                                         
    
sub movedir {
  my $srcdir = $_[0];
  my $destdir = $_[1];

  if (! -d $destdir) {
    mkdir($destdir);
  }

  opendir(DIR, $srcdir) or die "Can't open $srcdir: $!";
  my @files = grep {!/^\.+$/ } readdir(DIR);

  foreach my $file (@files) {
    my $old = "$srcdir/$file";
    move($old, $destdir) or die "Move $old -> $destdir failed: $!";
  }
  close(DIR);
}

####---------------------------------------------------------
#### Function for checking daemon/service
# ARGS: 2
# ARG1: daemon to be checked
# ARG2: num retries
# ARG3: boolean to enable error output or not.
sub check_service
{
    my $srv = $_[0];
    my $retries = $_[1];
    my $showtrace = $_[2];

    my $srv_running = FALSE;
    my $CRSCTL = crs_exec_path("crsctl");
    my $cmd = "$CRSCTL check $srv";
    my $grep_val;
    my @chk;
    my @cmdout;

    # for OHASD, we need to grep for CRS-4638
    # cannot use grep on Windows, customers are unlikely to have grep
    # on their systems

    # for CRS, we need to grep for CRS-4537
    if ($srv eq "ohasd") {
      $grep_val = "4638";
      $cmd = "$CRSCTL check has";
    }
    elsif ($srv eq "cluster") {
      my $node  = $CFG->HOST;
      $cmd      = "$CRSCTL check $srv -n $node";
      $grep_val = "4537";
      trace("Running $cmd");
    }
    elsif ($srv eq "css") {
      $grep_val = "4529";
    }
    else {
      # Add a check before the stat
      $cmd      = "$CRSCTL check resource $srv -init";
      @chk = system_cmd_capture1($showtrace, $cmd);
      # Return code of command is set on close, so capture now
      my $rc0 = shift @chk;
      if ($rc0 != 0) {
        trace("Check of service \"$_[0]\" failed\n".join("\n", @chk));
      }

      $cmd      = "$CRSCTL status resource $srv -init";
      $grep_val = "STATE=ONLINE";
    }

    # Wait for srv to start up
    while ($retries && ! $srv_running) {
      @chk = system_cmd_capture1($showtrace, $cmd);
      # Return code of command is set on close, so capture now
      my $rc = shift @chk;

      if ($grep_val) { @cmdout = grep(/$grep_val/, @chk); } # for OHASD

      # if scalar(@cmdout) > 0, we found the msg we were looking for
      if (($grep_val && scalar(@cmdout) > 0) ||
          (!$grep_val && $rc == 0)) {
        $srv_running = TRUE;
      }
      else {
        trace ("Checking the status of $srv");
        sleep (5);
        $retries--;
      }
    }

    # perform OSD actions
    s_check_service ($srv, $srv_running);

    return $srv_running;
}

####---------------------------------------------------------
#### Verify directory exists
# ARGS: 1
# ARG1 : Path to check
# @returns SUCCESS or $FAILURE
# static
sub check_dir {
  my $chkdirnm  = $_[0];
  if (!(defined($chkdirnm ))) {
    print_error(21);
    return FAILED;
  }
  if (!(-d $chkdirnm)) {
    print_error(22, $chkdirnm);
    return FAILED;
  }
  # not checking perms, since they may not be valid for root
  return SUCCESS;
}

####---------------------------------------------------------
#### Verify file exists
# ARGS: 1
# ARG1 : Path to check
# @returns SUCCESS or $FAILURE
# static
sub check_file {
  my $chkfilenm  = $_[0];
  if (!(defined($chkfilenm))) {
    print_error(23);
    return FAILED;
  }
  if (!(-f $chkfilenm)) {
    trace ("The setup file \"$chkfilenm\" does not exist");
    return FAILED;
  }
  # not checking perms, since they may not be valid for root
  return SUCCESS;
}

####---------------------------------------------------------
#### Function for starting daemon/service
# ARGS: 2
# ARG1: daemon to be started
# ARG2: user as whom daemon/service needs to be started
sub start_service
{
    my $srv  = $_[0];
    my $user = $_[1];
    my $status;

    # call OSD API
    $status = s_start_service ($srv);

    if ($status == SUCCESS) {
       trace("Started service '$srv'");
    } else {
       trace("Failed to start  service '$srv'");
    }

    return $status;
}

sub isODA
{
  my $OAKLIB = catfile("/opt", "oracle", "extapi", "64", "oak", "liboak.*.so");
  if (-f $OAKLIB)
  {
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}

sub isHAIPsupported
{
  if((! $CFG->SIHA) && s_is_HAIP_supported() &&
     (!is_dev_env() || ($ENV{'INSTALL_HAIP'} eq "true"))
    )
  {
    return TRUE;
  }
  else {
    return FALSE;
  }
}

sub isHAIPNonFatal
{
  if( ($ENV{'HAIP_NON_FATAL'} eq "true") || s_is_HAIP_NonFatal() )
  {
    return TRUE;
  }
  else 
  {
    return FALSE;
  }
}

sub isCRFSupported
{
   my $osysmond     = catfile ($CFG->ORA_CRS_HOME, "bin", "osysmond");
   my $osysmond_exe = catfile ($CFG->ORA_CRS_HOME, "bin", "osysmond.exe");

#In Production, configuration of CHM/OS depends on the user response during
#the install interview to configure MGMT_DB.
   if (!is_dev_env()  && ($CFG->defined_param('MGMT_DB')) &&
      ("false" eq lc($CFG->params('MGMT_DB'))))
   {
     trace ("Cluster Health Monitor has not been configured because ",
            "Grid Infrastructure Management Repository was not selected");

     return FALSE;
   }

   if ($^O =~ /win/i) {
      if (! (-e $osysmond_exe)) {
         trace ("Cluster Health Monitor osysmond not found");
         return FALSE;
      }

      return TRUE;
   }
   elsif ($^O =~ /linux|solaris|aix/i) {
      my $mach =`uname -m`;
      chomp($mach);

      if (! (-e $osysmond)) {
         trace ("Cluster Health Monitor osysmond not found");
         return FALSE;
      }
      elsif ($mach eq "s390x" || $mach eq "s390") {
         trace ("Cluster Health Monitor not supported on this platform");
         return FALSE;
      }

      return TRUE;
   }

   trace ("Cluster Health Monitor not supported on this platform");
   return FALSE;
}

sub checkServiceDown
#---------------------------------------------------------------------
# Function: Check if service is down
# Args    : 1 - service
# Returns : TRUE  if service is down
#           FALSE if service is up
#---------------------------------------------------------------------
{
   my $srv      = $_[0];
   my $crsctl   = crs_exec_path('crsctl');
   my $srv_down = FALSE;
   my @cmdout   = ();
   my ($grep_val, $cmd, $node);

   # for OHASD, we need to grep for CRS-4639
   if ($srv eq "ohasd") {
      $grep_val = "4639";
      $cmd      = "$crsctl check has";
   } elsif ($srv eq "cluster") {
      $grep_val = "4639";
      $cmd      = "$crsctl check cluster -n " . $CFG->HOST;
   } elsif ($srv eq "css") {
      $grep_val = "4639|4530";
      $cmd      = "$crsctl check css";
   }
   my @chk = system_cmd_capture($cmd);
   my $rc  = shift @chk;

   if ($grep_val) {
      @cmdout = grep(/$grep_val/, @chk);
   }

   # if scalar(@cmdout) > 0, we found the msg we were looking for
   if (($grep_val && scalar(@cmdout) > 0) ||
       (!$grep_val && $rc == 0)) {
      $srv_down = TRUE;
   }

   return $srv_down;
}

sub stop_resource
{
   my $CRSCTL       = crs_exec_path('crsctl');
   my @cmd          = ($CRSCTL, 'stop', 'resource', (@_));
   my $success      = TRUE;
   my @out          = system_cmd_capture_noprint(@cmd);
   my $status       = shift @out;

   if ($status == 0 || # stop successful
      scalar(grep(/CRS-0*2500/i, @out)) != 0) { # resource not up
      trace("Resource '@_' was not online but was successfully stopped");
   }
   elsif (check_service($_[0], 1)) { # check if service is still running
      trace("Stop of resource \"@_\" failed\n".join("\n", @out));
      print_error(116, $_[0]);
      $success = FALSE;
   }
   
   return $success;
}

sub stop_diskmon
{
  my $CRSCTL = crs_exec_path('crsctl');
  my $success = TRUE;

  # no diskmon in windows
  if ($CFG->platform_family eq "windows")
  {
    return TRUE;
  }

  if (check_service("ora.diskmon", 2)) {

    my @output = system_cmd_capture($CRSCTL,
                                    "stop",
                                    "resource",
                                    "ora.diskmon",
                                    "-init");
    my $status = shift @output;

    if ($status != 0 && !scalar(grep(/CRS\-2500/, @output)))
    {
        trace("Stop of resource \"ora.diskmon\" failed\n".join("\n", @output));
        print_error(116, "ora.diskmon");
        $success = FALSE;
    }
  }
  return $success;
}

=head2 configure_OCR

   Creates or updates OCR

=head3 Parameters

   None

=head3 Returns

  TRUE  - OCR configuration was     created or updated
  FALSE - OCR configuration was not created or updated

=cut

sub configure_OCR {
  my $success = TRUE;
  my $OCRCONFIGBIN = catfile ($CFG->ORA_CRS_HOME, "bin", "ocrconfig");
  my $status;
  my $lang_id = $CFG->params('LANGUAGE_ID');
  my $asmgrp = $CFG->params('ORA_ASM_GROUP');
  my $host = $CFG->HOST;

  # In case of cluster, 
  # Format/upgrade OCR using 'ocrconfig -upgrade ..' 
  # Populate keys using 'clscfg -install ...'
  if (!$CFG->SIHA)
  {
    my @runocrconfig = ("$OCRCONFIGBIN", "-upgrade",
                        $CFG->params('ORACLE_OWNER'),
                        $CFG->params('ORA_DBA_GROUP'));

    my $CLSCFGBIN = catfile ($CFG->ORA_CRS_HOME, "bin", "clscfg");
    my @runclscfg = ("$CLSCFGBIN", "-install",
                     "-h", $CFG->params('HOST_NAME_LIST'),
                     '-o', $CFG->ORA_CRS_HOME, 
                     '-g', $asmgrp);

    if ($CFG->CLSCFG_EXTRA_PARMS) {
      push @runclscfg, @{$CFG->CLSCFG_EXTRA_PARMS};
    }

    # abort if OCR is already configured
    if ((! $CFG->isRerun) && (! isReusedg()) && checkOCR($CFG->ORA_CRS_HOME))
    {
      trace("Existing OCR configuration found, aborting the configuration. Rerun configuration setup after deinstall");
      die(dieformat(428));
    }

    trace ("Creating or upgrading OCR keys");
    $status = system_cmd("@runocrconfig");
    if (0 != $status) {
      trace("ocrconfig -upgrade failed with status $status");
      print_error(157);
      $success = FALSE;
    }
    else {
      trace ("OCR keys are successfully populated");

      if (!s_reset_srvconfig()) {
         print_error(158);
         $success = FALSE;
      }
      else {
         #
         # clscfg - Initialize the Oracle Cluster Registry for the
         #          cluster. Should be done once per cluster install.
         #          Overwriting a configuration while any CRS daemon is
         #          running can cause serious issues.
         #
         trace("Executing clscfg");
         my @out = system_cmd_capture("@runclscfg");
         $status = shift @out;

         # Get true return value of clscfg (i.e. rc for spawned
         # process)
         if ((0 == $status) || ($status == 105 && (isReusedg() || $CFG->isRerun))) {
            trace ("Oracle Cluster Registry initialization completed");

            if ($CFG->CLSCFG_POST_CMD) {
               my @cmd = @{$CFG->CLSCFG_POST_CMD};
               system_cmd(@cmd);
            }
         }
         else {
            print_lines(@out);
            print_error(159);
            $success = FALSE;
         }
      }
    }
  }
  # In case of SIHA, 
  # Format OCR using 'clscfg -local'
  # Pin the node using 'crsctl pin css ...'
  else {
    # Create necessary configuration with 'clscfg -local'. Needed for 
    # compatibility with older DBs (10.x, 11.1). 
    my $clscfg = catfile ($CFG->ORA_CRS_HOME, "bin", "clscfg");
    my $cmdclscfg = "$clscfg -local -g $asmgrp";
    my $rc = system_cmd ($cmdclscfg);
    if ($rc != 0) {
       print_error(160, $cmdclscfg);
       exit;
    } else {
       trace ("Creating local-only OCR ($cmdclscfg) ... succeeded");
    }

    my $crsctl = crs_exec_path('crsctl');
    my $cmdcrsctl = "$crsctl pin css -n $host";
    $rc = system_cmd ($cmdcrsctl);
    if ($rc != 0) {
       print_error(161, $cmdcrsctl);
       exit;
    } else {
       trace ("Pin node ($cmdcrsctl) ... succeeded");
    }
  }

  return $success;
}

sub start_resource
{
  my $CRSCTL  = catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");
  my @cmd     = ($CRSCTL, 'start', 'resource', (@_));
  my $success = TRUE;

  my @out     = system_cmd_capture_noprint(@cmd);
  my $status  = shift @out;
  if ($status == 0) {
    if (scalar(@out) >0 ) {
      trace(join("\n>  ", ("Command output:", @out)),
            "\n>End Command output");
    }

    trace("Start of resource \"$_[0]\" Succeeded");
  }
  elsif (scalar(grep(/CRS-0*5702/i, @out)) != 0) { # resource already up
    trace("The resource \"$_[0]\" was already running");
  }
  elsif ($status != 0) 
  {
    if (scalar(@out) >0 ) {
      trace(join("\n>  ", ("Command output:", @out)),
            "\n>End Command output");
    }

    if (!check_service($_[0], 10))
    {
      trace("Start of resource \"$_[0]\" failed\n".join("\n", @out));
      print_error(115, $_[0]);
      $success = FALSE;
    }
    else
    {
      trace("The resource \"$_[0]\" is already running");
    }
  }

  return $success;
}

sub enable_HAIP
{
  trace("enable HAIP if supported");

  if(isHAIPsupported())
    {
      my $crsctl = catfile($CFG->ORA_CRS_HOME, 'bin', 'crsctl');
      my @cmd    = ($crsctl, 'modify', 'res', 
                    'ora.cluster_interconnect.haip', '-attr', 
                    "\"ENABLED=1\"", '-init');
      my $status;
      $status = system_cmd(@cmd);
      if ( 0 != $status ) {
        trace("Failed to enable HAIP startup: $status");
      }

      if ( ! start_resource("ora.cluster_interconnect.haip", "-init") ) {
        trace("Failed to startup HAIP");

        # if HAIP is not considered fatal on the platform then disable
      # further startup attempts
      if( isHAIPNonFatal() )
      {
        my @cmd    = ($crsctl, 'modify', 'res', 
                      'ora.cluster_interconnect.haip', '-attr', 
                      "\"ENABLED=0\"", '-init');
        $status = system_cmd(@cmd);
        if( 0 != $status )
        {
          trace("Failed to disable HAIP startup: $status");
        }
      }
      else
      {
        die(dieformat(246));
      }
    }
  }
}

# TODO: We don't need this when upgrade stops using this
sub start_clusterware {
   my $level              = $_[0];
   my $status             = SUCCESS;
   my %stack_start_levels =
      (START_STACK_MDNSD  => 'Oracle clusterware daemons up to MDNSD',
       START_STACK_GPNPD  => 'Oracle clusterware daemons up to GPNPD',
       START_STACK_GIPCD  => 'Oracle clusterware daemons up to GIPCD',
       START_STACK_CRF    => 'Oracle clusterware daemons up to IPD/OS',
       START_STACK_CTSSD  => 'Oracle clusterware daemons up to CTSSD',
       START_STACK_CSSD   => 'Oracle clusterware daemons up to CSSD',
       START_STACK_ASM    => 'Oracle clusterware daemons up to ASM',
       START_STACK_ALL    => 'the Oracle clusterware stack'
      );

   trace ("Starting", $stack_start_levels{$level});

   if (($level < START_STACK_EVMD  || start_resource("ora.evmd", "-init")) &&
       ($level < START_STACK_MDNSD || start_resource("ora.mdnsd", "-init")) &&
       ($level < START_STACK_GPNPD || (start_resource("ora.gpnpd", "-init") &&
                                       wait_for_gpnpd_start()))  &&
       ($level < START_STACK_GIPCD || start_resource("ora.gipcd", "-init")) &&
       ($level < START_STACK_CRF   || !isCRFSupported() || 
                                      start_resource("ora.crf", "-init")) &&
       ($level < START_STACK_CSSD  || CSS_start_clustered()) &&
       ($level < START_STACK_CTSSD || start_resource("ora.ctssd", "-init", 
                                      "-env", "USR_ORA_ENV=CTSS_REBOOT=TRUE")) &&
       ($level < START_STACK_ASM   || !$CFG->ASM_STORAGE_USED  || # if ASM used, start it
                                      start_resource("ora.asm", "-init"))  &&
       ($level < START_STACK_CRSD || $level < START_STACK_ALL ||
        start_resource("ora.crsd", "-init")))
   {
      trace ("Successfully started requested Oracle stack daemons");
   }
   else {
      print_error(117);
      $status = FAILED;
   }

   return $status;
}

=head2 wait_for_stack_start

  Wait for the stack to start up

=head3 Parameters

  Number of chcks to see if the stack is up, made every 5 seconds

=head3 Returns

  SUCCESS  Stack is up
  FAILED   Stack is not up

=head3 Usage


=cut

sub wait_for_stack_start {
  # Wait until the daemons actually start up
  my $is_up = FALSE;
  my $retries = shift;
  my $crsctl = crs_exec_path('crsctl');
  my @output;
  my $rc;

  # Complete success. This is the last node of the install.
  # Wait for CRSD to start up
  while ($retries) {
    @output = system_cmd_capture($crsctl, 'check', 'crs');
    $rc     = shift @output;
    if ($rc == 0) {
      $is_up = TRUE;
      last;
    }

    trace ("Waiting for Oracle CRSD to start");
    sleep (5);
    $retries--;
  }

  if ($is_up) {
    #Also wait for check cluster to work.
    if (!check_service("cluster", 120))
    {
       die(dieformat(251));
    }
    trace ("Oracle CRS stack installed and running");
   
  } else {
    print_error(120);
    exit 1;
  }

  return $is_up;
}

####---------------------------------------------------------
#### Function for copying onc.config to Oracle 10g home
# ARGS : 1
# ARG1 : Oracle CRS home
sub copyONSConfig
{
    my $crshome = $_[0];

    if (!$crshome) {
        print_error(4);
        return FAILED;
    }

    if (!(-d $crshome)) {
        print_error(5, $crshome);
        return FAILED;
    }

    trace ("Oracle CRS home = " . $crshome);
    trace ("Copying ONS config file to 10.2 CRS home");

    my $OLSNODESBIN = catfile ($crshome, "bin", "olsnodes");
    if (-x $OLSNODESBIN) {
        open (OLSNODES, "$OLSNODESBIN -l|");
        my $NODE_NAME = <OLSNODES>;
        close (OLSNODES);
        my $OCRDUMPBIN = catfile ($crshome, "bin", "ocrdump");
        if (-x $OCRDUMPBIN) {
            if ($CFG->platform_family eq "windows") {
                open (OCRDUMP, "$OCRDUMPBIN -stdout -keyname
                      CRS.CUR.ora!$NODE_NAME!ons.ACTION_SCRIPT |");
            }
            else {
                open (OCRDUMP, "$OCRDUMPBIN -stdout -keyname
                      'CRS.CUR.ora!$NODE_NAME!ons.ACTION_SCRIPT'|");
            }

            my @output = <OCRDUMP>;
            close (OCRDUMP);
            my $txt = grep (/ORATEXT/, @output);
            my ($key, $ONS_OH) = split (/:/, $txt);
            $ONS_OH =~ s!/bin/racgwrap!!g;
            $ONS_OH =~ s/^ //g;
            ## checking if ONS resource is configured
            if ($ONS_OH) { 
                ##ONS resource is configured
                my $ONSCONFIG = catfile($ONS_OH, "opmn", "conf", "ons.config");
                my $ONSCONFIG_CH =
                    catfile($ONS_OH, "opmn", "conf", "ons.config");
                if (-f $ONSCONFIG) {
                    ##The ons.config file exists at source location
                    copy ($ONSCONFIG_CH, "$ONSCONFIG_CH.orig");
                    copy ($ONSCONFIG, $ONSCONFIG_CH); 
                    trace ("$ONSCONFIG was copied successfully to " .
                           $ONSCONFIG_CH);
                }
            }
        }
    }

    return SUCCESS;
}

=head2 get_crs_version

  Gets parsed version numbers of active CRS version.
  Version is a result of "crsctl query crs activeversion" command.
  Stack (CRS) must be up for this to succeed.

=head3 Parameters

  string with crsctl home location. If undef, then current home is used.
   
=head3 Returns

=head4 returns an array of version numbers major to minor. 
       If error occurred, all numbers will be 0. Error will be printed.       

=cut

sub get_crs_version 
{
   my $home = $_[0];
   my @ver  = (0, 0, 0, 0, 0);
   my ($crsctl);
   my $verstring;

   if (! $home) {
      $crsctl = crs_exec_path('crsctl');
   } else {
      $crsctl = catfile($home, 'bin', 'crsctl' );
   }

   if (isOCRonASM()) {
     trace("setting ORAASM_UPGRADE to 1");
     $ENV{'ORAASM_UPGRADE'} = "1";
   }

   # run "crsctl query crs activeversion" -- stack must be up
   # Example output:
   # Oracle Clusterware active version on the cluster is [11.2.0.0.2]
   my @cmd = ($crsctl, 'query', 'crs', 'activeversion');
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;

   # if succeeded, parse to ver numbers, output must be a single line,
   # version is 5 numbers, major to minor (see above)
   if ($rc == 0) {
      $verstring = getVerInfo($out[0]);
      trace( "Got CRS active version: ".join('.', $verstring) );
   }
   else {
      trace ("@cmd ... failed rc=$rc with message:\n @out \n");
      trace ("Getting active version from OCR");
      $verstring = get_OldCrsVersionfromOCR();
      trace("old crs active version retreived from OCR is $verstring");
   }
   @ver = split(/\./, $verstring);
   trace( "Got CRS active version: ".join('.', @ver) );
   return @ver;

}

sub isCVUConfigured
#-------------------------------------------------------------------------------
# Function: Find if CVU resource is already configured
# Args    : oldCrsHome
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
{
  my $status = SUCCESS;
  my @out;
  my $rc;
  my $oldCrsHome = shift;

  trace("Invoking srvctl config cvu");

  my $srvctl =  catfile($oldCrsHome, 'bin', 'srvctl');

  @out = system_cmd_capture($srvctl, "config", "cvu");
  $rc = shift @out;

  if ($rc != 0) {
    trace("CVU resource does not exist. output is @out");
    $status = FAILED;
  } else  {
    trace("CVU resource  exist. output is @out");
  }

  return $status;
}

sub get_CVU_checkInterval
#-------------------------------------------------------------------------------
# Function: get CVU resources check_interval value
# Args    : oldCrsHome
# Returns : CVU check_interval value if successful. 0 in case of failure
#-------------------------------------------------------------------------------
{
   my $checkIntervalStr = "0";
   my $oldCrsHome = shift;
   my $outline;
   my @out;
   my $rc;

   trace("Invoking srvctl config cvu");

   my $srvctl =  catfile($oldCrsHome, 'bin', 'srvctl');

   @out = system_cmd_capture($srvctl, "config", "cvu", "-S", "1");
   $rc = shift @out;

   if ($rc != 0) {
      trace("CVU resource config could not be obtained. output is @out");
   } else  {
      trace("CVU resource  exist. output is @out");
      # Example output:
      # #@=result[0]: res_name={ora.cvu} check_interval={360}
      $outline = $out[0];
      $outline = trim($outline);
      $outline =~ m/\{(\d*)\}*$/;
      if ($1 && $1 ne "") {
         $checkIntervalStr = $1;
      }
   }

   trace("check interval is : [$checkIntervalStr]");
   return $checkIntervalStr;
}
                             
sub isRolling
{
  my $isrolling = TRUE;
  if ($CFG->params('ISROLLING') =~ m/false/i) {
      $isrolling = FALSE;
   }
  trace("Rolling upgrade is set to $isrolling");
  return $isrolling;
}

sub getVerInfo
#-------------------------------------------------------------------------------
# Function: Get the the Version from the String Passed
# Args    : VerString
# Returns : VerInfo 
#-------------------------------------------------------------------------------
{
   my $verstring = $_[0];
   my @verarray = (0, 0, 0, 0, 0);
   my $verinfo;
   trace("Version String passed is: [$verstring]");
   if ($verstring)
   {
      $verstring =~ m/\[?(\d*)\.(\d*)\.(\d*)\.(\d*)\.(\d*)\]?.*$/;
      @verarray = ($1, $2, $3, $4, $5);
      $verinfo = join('.',@verarray);
      trace("Version Info returned is : [$verinfo]");
   }
   else
   {
      trace("Null Version String is Passed to getVerInfo");
   }

   return $verinfo;
}

sub get_OldCrsVersionfromOCR
{
   trace("Getting old crs active verison  from ocrdump");

   my $OCRDUMPBIN  = catfile ($CFG->ORA_CRS_HOME, 'bin', 'ocrdump');
   if ($CFG->platform_family eq "windows") {
      open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
              "SYSTEM.version.activeverison |");
   }
   else {
      open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
                  "'SYSTEM.version.activeverison'|");
   }
   my @output = <OCRDUMP>;
   close (OCRDUMP);

   trace("ocrdump output for active version is @output");
   my @txt = grep (/ORATEXT/, @output);
   my ($key, $actver) = split (/:/, $txt[0]);

   trace ("key is $key");
   trace ("activeversion is $actver");
   return $actver;
                         
}

sub waitForVipRes 
#-------------------------------------------------------------------------------
# Function: Wait for VIP resource to exist
# Args    : none
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
{
   my $host       = $CFG->HOST;
   my $srvctl     = crs_exec_path('srvctl');
   my $retries    = 12;
   my $vip_exists = FALSE;
   my ($rc, @output);

   while ($retries) {
      @output = system_cmd_capture($srvctl, 'status', 'vip', '-n', "$host");
      $rc     = shift @output;

      trace("rc=$rc output=@output");
      my @cmdout = grep(/(^PRKO-2165)/, @output);
      if ($rc == 0 && scalar(@cmdout) == 0) {
         $vip_exists = TRUE;
         last;
      }

      trace ("Waiting for VIP resource");
      sleep (5);
      $retries--;
   }

   if ($vip_exists) {
      trace ("VIP resource exists");
   } 
   else {
      trace ("Timed out waiting for VIP resource");
   }

   return $vip_exists;
}

sub isFirstNodeToStart 
######################################################################
# Returns:
#   FALSE   if node is not first to start
#   TRUE    if node is     first to start
######################################################################
{
   my $isFirst = FALSE;

   # A rim node can never be a first node. The first node is always a hub node.
   if (isRimNode())
   {
      return FALSE;
   }

   # Get the list of nodes that have started
   my $olsnodes = catfile($ENV{'ORA_CRS_HOME'}, 'bin', 'olsnodes');
   trace("Running $olsnodes");
   open ON, "$olsnodes |";
   my @olsnodes = (<ON>);
   close ON;

   chomp @olsnodes;

   trace("Nodes returned by olsnodes: @olsnodes");
   if (scalar(@olsnodes) == 1)
   {
      $isFirst = TRUE;
   }

   return $isFirst;
}

sub add_Nodeapps
#-------------------------------------------------------------------------------
# Function: Add nodeapps for static IP & DHCP
# Args    : [0] upgrade_opt
#           [1] nodevip
#           [2] DHCP_flag
#           [3] nodes_to_add
#           [4] nodes_to_start
# Returns : TRUE  if success
#           FALSE if failed
#           nodes_to_start - list of nodes to start
#-------------------------------------------------------------------------------
{
   my $upgrade_opt        = shift;
   my $nodevip_ref        = shift;
   my $isDHCP             = shift;
   my $nodes_to_add_ref   = shift;
   my $nodes_to_start_ref = shift;

   trace ("adding nodeapps...");
   trace ("upgrade_opt=$upgrade_opt");
   trace ("nodevip=@$nodevip_ref");
   trace ("DHCP_flag=$isDHCP");
   trace ("nodes_to_add=@$nodes_to_add_ref");

   if (0 == scalar(@$nodes_to_add_ref))
   {
     trace("nodes_to_add is null");
     return FALSE;
   }

   my $srvctl_func = \&srvctl;
   if ((($^O eq "linux") || ($^O eq "solaris")) && ($CFG->AUTO))
   {
     trace("Call srvctl_tty");
     $srvctl_func = \&srvctl_tty; 
   }

   my $config_nodeapps  = catfile ($CFG->ORA_CRS_HOME, "bin",
                                   "srvctl config nodeapps");
   my $vip_exists       = FALSE;
   my $success          = TRUE;
   my $run_as_owner     = FALSE;
   my @output;

   if ($isDHCP) {
      trace ("add nodeapps for DHCP");
      my $node    = $$nodes_to_add_ref[0];
      push @$nodes_to_start_ref, $node;

      # Currently we don't support multiple public subnets. The installer should
      # be smart enough not to allow user to select more than 1 public subnet.
      my @subnets = getSubnets ($CFG->params('NETWORKS'), 'public');
      my $subnet = shift (@subnets);
      my $nodevip = shift (@$nodevip_ref);
      # substitute AUTO w/ subnet
      $nodevip    =~ s/AUTO/$subnet/;

      my $status = srvctl($run_as_owner,
                          "add nodeapps -S \"$nodevip\" $upgrade_opt");
      if (${status}) {
         trace ("add nodeapps -S $nodevip on node=$node ... passed");
      } else {
         trace("add nodeapps -S $nodevip on node=$node ... failed");
         print_error(180, "srvctl add nodeapps -S $nodevip $upgrade_opt", $status);
         $success = FALSE;
      }

      return $success;
   }

   # add nodeapps for STATIC IP
   trace("add nodeapps for static IP");

   # The following  check is only valid in case of an addnode scenario
   if (! $CFG->UPGRADE) {
      trace("Running srvctl config nodeapps to detect if VIP exists");
      open OPUT, "$config_nodeapps |";
      @output = grep(/(^PRKO-2312|^PRKO-2331|^PRKO-2339)/, <OPUT>);
      close OPUT;
      if (scalar(@output) == 0) {
         trace ("vip exists");
         $vip_exists = TRUE;
      } else {
         trace ("output=@output");
      }
   }

   foreach my $node (@$nodes_to_add_ref) {
      $node       =~ tr/A-Z/a-z/; #convert to lowercase
      my $nodevip = shift (@$nodevip_ref) or die(dieformat(275));
      my @txt     = grep (/$node/, @output);
      my $status;
      my $cmd;

      if (scalar(@txt) == 0) {   # nodeapps is not yet added on this node
         if ($vip_exists) {
            $cmd = "add vip -n $node -k 1 -A $nodevip";
            if (isBigCluster() && isRimNode())
            {
              trace("No need to add vip on rim node");
              $status = TRUE;
            }
            else
            {
              $status = &$srvctl_func($run_as_owner,
                               "add vip -n $node -k 1 -A \"$nodevip\" " .
                               "$upgrade_opt");
            }
         } else {
            $vip_exists = TRUE;
            if ($CFG->UPGRADE){
               my($onslocport, $onsremport) = get_ons_port($node);
               $cmd = "add nodeapps -n $node -l $onslocport " .
                      "-r $onsremport -A $nodevip";
               $status = &$srvctl_func($run_as_owner,
                         "add nodeapps -n $node -l $onslocport -r $onsremport " .
                         "-A \"$nodevip\" $upgrade_opt");
            } else {
               $cmd = "add nodeapps -n $node -A $nodevip";
               $status = &$srvctl_func($run_as_owner,
                                "add nodeapps -n $node -A \"$nodevip\" " .
                                "$upgrade_opt");
            }
         }

         if (${status}) {
            push @$nodes_to_start_ref, $node;
            trace ("$cmd on node=$node ... passed");
         } else {
            trace("$cmd on node=$node ... failed");
            print_error(180, $cmd, $status);
            $success = FALSE;
         }
      }
   }

   trace ("nodes_to_start=@$nodes_to_start_ref");
   return $success;
}

sub getSubnets
#---------------------------------------------------------------------
# Function: Get subnets from $NETWORKS of the network type $net_type
# Args    : [0] $NETWORKS
#           [1] $net_type
# Returns : Array of subnets.
#---------------------------------------------------------------------
{
   my $networks = $_[0];
   my $net_type = $_[1];
   my ($eth, $txt, $subnet, @subnetList);

   if ($networks =~ /\b$net_type\b/) {
      my @network_ifs = split (/,/, $networks);

      foreach my $network_if (@network_ifs) {
         if ($network_if =~ /\b$net_type\b/) {
            # strip out "eth*" and ":$net_type"
            ($eth, $txt) = split (/\//, $network_if);
            ($subnet, $txt) = split (/:$net_type/, $txt);

            push @subnetList, $subnet;
         }
      }
   }

   return @subnetList;
}

sub get_ons_port
#---------------------------------------------------------------------
# Function: Get the ONS port used by the old version crs
#---------------------------------------------------------------------
{
   my $node = $_[0];
   my ($home, $ONSCONFFILE, $Name, $portnum, $useocr, $localport, $remoteport,
       $locport, $remport, $cmd, $ocrkey, $line);
   my $idx = 0;
   my @buf2;
 
   # if version is 10.1, use dbhome where ons is configured. Otherwise, use OLD_CRS_HOME.
   my @old_version = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   if ($old_version[0] eq "10" &&
      $old_version[1] eq "1") {
      $home = get101viphome();
   }
   else
   {
      $home = $CFG->OLD_CRS_HOME;
   }
 
   $ONSCONFFILE = catfile( $home, 'opmn' , 'conf', 'ons.config');
   trace("The ons conf file location is: $ONSCONFFILE");
   open(FONS, $ONSCONFFILE) or
        trace("Could not  open \"$ONSCONFFILE\": $!");

   while(<FONS>) {
     if(/^useocr\=on\b/i) { $useocr=$_;  }
     if(/^remoteport\b/i) { $remoteport=$_;  }
     if(/^localport\b/i)  { $localport=$_;  }
   }

   close (FONS);
   #get the remote port
   if (defined($useocr)) {
      trace("useocr is on. get the remote port from OCR");
      $cmd = catfile ($CFG->ORA_CRS_HOME, 'bin', 'ocrdump' );
      $ocrkey = "DATABASE.ONS_HOSTS";

      my @args = ($cmd, '-stdout', '-keyname', $ocrkey);
      my @out = system_cmd_capture(@args);

      if ($CFG->DEBUG) { trace("ocrdump output: @out"); }

      my $rc  = shift @out;
      foreach $line (@out) {
         if ($line =~ m/DATABASE\.ONS_HOSTS\.$node.*\.PORT\]/i) {
            @buf2 = $out[$idx+1];
            last;
         }
         
         $idx++;
      }

      if ($CFG->DEBUG) { trace("ocrdump output for port information is: @buf2"); }

      if (scalar(@buf2) != 0) {
        ($Name, $portnum) = split(/:/, $buf2[0]);
      }
         $remport = trim($portnum);
   }
   else 
   {
    if (defined($remoteport)) {
        ($Name, $portnum) = split(/=/, $remoteport);
        $remport = trim($portnum);
    }
   }

   #always get the localport from ons.config if present
   if (defined($localport)) {
     ($Name, $portnum) = split(/=/, $localport);
     $locport = trim($portnum);
   }

   #set 11.2 default values for ons port
   if (! $locport) {
       trace("setting default port  for ons localport");
       $locport = "6100";
   }
   if (! $remport) {
       trace("setting default port  for ons remoteport");
       $remport = "6200";
   }

   trace("Local port=$locport");
   trace("Remote port=$remport");

   return ($locport, $remport);
}

#For 10.1, get the oracle home location where the VIP
#resources are configured.
sub get101viphome
{
  my $host    = $CFG->HOST;
  my $ocrdump = catfile ($CFG->params('ORACLE_HOME'), 'bin', 'ocrdump');

  # get ons.ACTION_SCRIPT keyname
  if ($CFG->platform_family eq "windows") {
     open (OCRDUMP, "$ocrdump -stdout -keyname " .
           "CRS.CUR.ora!$host!ons.ACTION_SCRIPT |");
  }
  else {
     open (OCRDUMP, "$ocrdump -stdout -keyname " .
           "'CRS.CUR.ora!$host!ons.ACTION_SCRIPT' |");
  }

  my @output = <OCRDUMP>;
  close (OCRDUMP);

  if ($CFG->DEBUG) { trace("get101viphome:ocrdump = @output"); }

  # get vip home
  my @txt = grep (/ORATEXT/, @output);
  my ($key, $vip_home) = split (/: /, $txt[0]);
  $vip_home =~ s!/bin/racgwrap!!g;
  $vip_home =~ s/^ //g;
  chomp($vip_home);

  if ($CFG->DEBUG) { trace("get101viphome:ons home = $vip_home"); }

  return $vip_home;
}

sub configFirstNode
#---------------------------------------------------------------------
# Function: Configure first node
# Args    : [0] DHCP_flag
#           [1] nodes_to_start
# Returns : TRUE  if success
#           FALSE if failed
#---------------------------------------------------------------------
{
   my $DHCP_flag          = shift;
   my $nodes_to_start_ref = shift;
   my $run_as_owner = TRUE;

   trace ("Configuring first node");
   trace ("DHCP_flag=$DHCP_flag");
   trace ("nodes_to_start=@$nodes_to_start_ref");

   if (0 == scalar(@$nodes_to_start_ref))
   {
     trace("nodes_to_start is null");
     return FAILED;
   }

   #set the network interface - Bug 9243302
   setNetworkInterface() || return FAILED;

   if (!(add_GNS() &&
         add_scan() &&
         add_scan_listener() &&
         add_J2EEContainer() &&
         add_rim_listener()))
   {
      return FAILED;
   }

   if (($CFG->params('MGMT_DB') =~ m/true/i) && !add_mgmt_db_listener())
   {
      return FAILED;
   }

   # Do not create asm or diskgroup resources on a ASM client cluster
   if (($CFG->params('ASM_UPGRADE') =~ m/false/i) && (! isASMExists()) && !isFarASM())
   {
      trace("Prior version ASM does not exist , Invoking add asm");
      add_ASM();  # add ora.asm
      if ($CFG->ASM_STORAGE_USED)
      {
         createDiskgroupRes() || return FAILED;  # add disk group resource, if necessary
      }
    }

   add_CVU("0") || return FAILED;

   if (start_Nodeapps($DHCP_flag, \@$nodes_to_start_ref) &&
       start_GNS() &&
       start_scan() &&
       start_scan_listener())
   {
       start_J2EEContainer() || return FAILED;
   } else
   {
       return FAILED;
   }

   start_CVU() || return FAILED;

   return SUCCESS;
}

sub setNetworkInterface
{
  my $success = TRUE;
  my $oifcfg = catfile($CFG->params('ORACLE_HOME'), 'bin', 'oifcfg');
  my $networks = oifcfgNetworks();
 
  my @out = system_cmd_capture($oifcfg, "setif -global", $networks);
  my $rc = shift @out;

  if ($rc == 0)
  {
     trace("$oifcfg setif -global $networks successful");
  } else {
     trace("$oifcfg setif -global $networks failed rc = $rc");
     print_error(180, "$oifcfg setif -global $networks", $rc);
     $success = FALSE;
  }

  return $success;
}


=head2 oifcfgNetworks

  Converts NETWROKS to a list that can be accepted by oifcfg  

  E.g., from 
  NETWORKS="eth0"/1.2.3.4:public,"eth1"/2.3.4.5:cluster_interconnect,"eth1"/2.3.4.5:asm

  to
  "eth0"/1.2.3.4:public "eth1"/2.3.4.5:cluster_interconnect,asm

=head3 Parameters

  None  

=head3 Returns

  A space-separated list passed to oifcfg

=cut

sub oifcfgNetworks
{
  my @networks = split(/,/, $CFG->params('NETWORKS'));
  my %nets;
  my $netSpecs;

  foreach my $elem (@networks)
  {
     # In a test environment oifcfg setting same interface to pvt and public
     # throws an error. For this reason a check is placed to skip setting
     # public interface in dev environment only.
     # oifcfg setif -global eth2/10.232.192.0:public,cluster_interconnect,asm
     # PRIF-53: Invalid type combination specified for interface [eth2]
     # Remove this check after Bug #13806475 is fixed.
     #next if (( $elem !~ /cluster_interconnect/i) && ( is_dev_env() ) && ( $CFG->params('NETWORKS') =~ /asm/ ));
     # Look for the last colon to separate <interface>/<subnet> and 
     # <interface_type>
     my @net = split(/:(public|cluster_interconnect|asm)/, $elem);
     my $type = $nets{$net[0]};
     if ($type)
     {
       # Commbine multile interface types for the same <interface>/<subnet> 
       # using commas as separator
       $type = $type.",".$net[1];
       $nets{$net[0]} = $type;
     }
     else
     {
       $nets{$net[0]} = $net[1];
     } 
  }

  my @array;
  while (my($key, $value) = each % nets)
  {
    my $spec = $key.":".$value;
    push(@array, $spec);    
  }

  # Create a space-separated list that is passed to oifcfg
  if (scalar(@array) > 0)
  {
    $netSpecs = join(" ", sort(@array));
  }

  return $netSpecs;
}

=head2 compact_instlststr  

  Converts a installer-prepared list containing duplicates to a compact 
  version, which is required by sub instlststr_to_gpnptoolargs

  E.g., from
  "eth0"/1.2.3.4:public,"eth1"/2.3.4.5:cluster_interconnect,"eth1"/2.3.4.5:asm

  to
  "eth0"/1.2.3.4:public,"eth1"/2.3.4.5:cluster_interconnect|asm

=head3 Parameters

  A installer-prepared list string 

=head3 Returns

  A compact version of install-style network info list that is passed to
  instlststr_to_gpnptoolargs()

=cut

sub compact_instlststr 
{
  my $pNets = $_[0];
  $pNets = ($pNets) ? ($pNets) : ($CFG->params('NETWORKS'));

  trace("Ready to parse: $pNets");
  my @networks = split(/,/, $pNets);
  my %nets;
  my $netSpecs;
  
  foreach my $elem (@networks)
  {
     # Look for the last colon to separate <interface>/<subnet> and 
     # <interface_type>

     # Some srgs/lrgs populate NETWORKS with compact format, e.g., in srgrsc:
     # NETWORKS=eth0/10.232.168.0:cluster_interconnect|public
     # Do nothing and just return the original one in this case
     if ($elem =~  m/(^.+)[:]((public|cluster_interconnect|asm)([|](public|cluster_interconnect|asm))+)$/)
     {
       trace("The original one is already a compact version");
       return $pNets;
     }

     my @net = split(/:(public|cluster_interconnect|asm)/, $elem);
     my $type = $nets{$net[0]};
     if ($type) 
     {
       # Commbine multile interface types for the same <interface>/<subnet> 
       # using '|' as separator
       $type = $type."|".$net[1];
       $nets{$net[0]} = $type;
     }
     else
     {
       $nets{$net[0]} = $net[1];
     }
  }

  my @array;
  while (my($key, $value) = each % nets)
  {
    my $spec = $key.":".$value;
    push(@array, $spec);
  }

  # Create a comma-separated list that is passed to instlststr_to_gpnptoolargs() 
  if (scalar(@array) > 0)
  {
    $netSpecs = join(",", sort(@array));
  }

  return $netSpecs;
}

sub start_CVU
#------------------------------------------------------------------------------
# Function:  Start the cvu resource
# Args    :  none
#-------------------------------------------------------------------------------
{
   my $run_as_owner = TRUE;
   my $status = srvctl($run_as_owner, "start cvu");

   if (${status}) {
      trace ("start cvu ... success");
   } else {
      print_error(112);
      return FALSE;
   }

   return TRUE;
}

sub remove_checkpoints
{
   my $host = tolower_host();
   my $ckpt_filename = "ckptGridHA_".$host.".xml";
   my $ckpt_file = catfile ($CFG->params('ORACLE_BASE'), "crsdata", $host, "crsconfig",
                            $ckpt_filename);
   trace ("Removing the checkpoint file $ckpt_file");

   s_remove_file ($ckpt_file);
}

sub isASMExists
#-------------------------------------------------------------------------------
# Function:  Check if ASM exists
# Args    :  none
# Returns :  TRUE  if     exists
#            FALSE if not exists
#-------------------------------------------------------------------------------
{
   my $crs_home = $CFG->ORA_CRS_HOME;
   my $host     = $CFG->HOST;
   my $crs_stat = catfile ($crs_home, 'bin', 'crs_stat');

   open (CRSSTAT, "$crs_stat |");

   # temporarely using crs_stat to find pre 11.2 ASM
   # grep "ora.$host*asm"
   my @txt = grep /ora.$host.*asm/, <CRSSTAT>;

   close (CRSSTAT);

   if (scalar(@txt) == 0) {
      trace ("check ASM exists done and ASM does not exist");
      return FALSE;
   }

   return TRUE;
}

sub add_ASM 
{
   my $run_as_owner = TRUE;
   my $asmPwdFile = $CFG->ASM_PWD_FILE;

   if (($CFG->ASM_STORAGE_USED) && (! $asmPwdFile))
   {
     die(dieformat(379));
   }

   if (isLegacyASM())
   {
     trace("add asm ...");

     my $cmd;
     if ($CFG->ASM_STORAGE_USED)
     {
       trace("The path for ASM password file is $asmPwdFile");
       $cmd = "add asm -pwfile ${asmPwdFile}";
     }
     else
     {
       $cmd = "add asm";
     }

     srvctl($run_as_owner, $cmd) || die(dieformat(180, "srvctl add asm", 0));
   }

   if (isNearASM())
   {
     trace("add asm -flex ...");
     srvctl($run_as_owner, "add asm -flex -pwfile ${asmPwdFile}") ||
       die(dieformat(371));

     trace("add asm listeners");
     my $aux; # used for the listeners' names
     my @subnets = getSubnets($CFG->params('NETWORKS'), 'asm');

     foreach my $subnet (@subnets) {
       srvctl($run_as_owner, "add listener -asmlistener -listener " .
                             "asmnet" . ++$aux . "lsnr -subnet $subnet") ||
          die(dieformat(180, "srvctl add listener -asmlistener -listener " .
                             "asmnet${aux}lsnr -subnet $subnet", 0));
     }

   } # isNearASM
}

sub createDiskgroupRes
#---------------------------------------------------------------------
# Function: Create and start ASM diskgroup resource on all nodes
# Args    : none
# Returns : TRUE  if success
#           FALSE if failed
#---------------------------------------------------------------------
{
   my $success = TRUE;
   my $cmd;

   trace ("Adding ASM diskgroup resource");

   # convert ASM_DISK_GROUP to upper-case
   my $crsctl = crs_exec_path('crsctl');
   my $ASM_DISK_GROUP = uc($CFG->params('ASM_DISK_GROUP'));
   if ($ASM_DISK_GROUP =~ /\$/) {
      # if diskgroup contains '$', put single-quotes around it
      quoteDiskGroup($ASM_DISK_GROUP);
      $cmd = "$crsctl create diskgroup '$ASM_DISK_GROUP'";
   }
   else {
      $cmd = "$crsctl create diskgroup $ASM_DISK_GROUP";
   }

   my $status = run_as_user ($CFG->params('ORACLE_OWNER'), $cmd);

   if ($status == 0) {
      trace ("create diskgroup $ASM_DISK_GROUP ... success");
   } else {
      print_error(182, $ASM_DISK_GROUP);
      return FALSE;
   }

   trace ("Successfully created disk group resource");

   # since diskgroup rescource is successfully added on the lastnode
   # we need to start diskgroup rescource on other nodes
   # get the local node
   my $olsnodes = catfile($CFG->ORA_CRS_HOME, 'bin', 'olsnodes -l');
   open (OLSNODES, "$olsnodes |") or die(dieformat(180, $olsnodes, $!));
   my @output = (<OLSNODES>);
   close OLSNODES;
   chomp @output;
   my $local_node = $output[0];

   # get the list of all nodes
   $olsnodes = catfile($CFG->ORA_CRS_HOME, 'bin', 'olsnodes');
   open (OLSNODES, "$olsnodes |") or die(dieformat(180, $olsnodes, $!));
   my @nodes = (<OLSNODES>);
   close OLSNODES;

   # build node_list from olsnodes, except for local node
   my $node_list = "";
   foreach my $node (@nodes) {
      chomp $node;
      if ($node ne $local_node) {
         if ($node_list ne "") {
            $node_list = $node_list . ",";
         }

         $node_list = $node_list . $node;
      }
   }

   # node_list eq "" means 1-node install
   if ($node_list ne "") {
      # start diskgroup on all nodes
      my $run_as_owner = FALSE;
      $status = srvctl($run_as_owner,
                       "start diskgroup -g $ASM_DISK_GROUP -n \"$node_list\" ");

      if (${status}) {
         trace ("start diskgroup resource ... success");
      } else {
         print_error(109);
         return FALSE;
      }
   }

   return $success;
}

sub quoteDiskGroup
#-------------------------------------------------------------------------------
# Function: Check if asm disk group contains '$'
# Args    : diskgroup
# Returns : diskgroup w/ '\' character
#-------------------------------------------------------------------------------
{
   if ($_[0]) {
      $_[0] =~ s/\$/\\\$/g;
   }
}

sub add_GNS
#---------------------------------------------------------------------
# Function: Add GNS
# Returns : TRUE  if success
#           FALSE if failed
#---------------------------------------------------------------------
{
   if ($CFG->params('GNS_CONF') ne "true") {
      trace ("GNS is not to be configured - skipping");
      return TRUE;
   }

   my $srvctl_func = \&srvctl;
   if ((($^O eq "linux") || ($^O eq "solaris")) && ($CFG->AUTO))
   {
     trace("Call srvctl_tty");
     $srvctl_func = \&srvctl_tty; 
   }

   my $run_as_owner = FALSE;
   my ($gns_type, $credentials, $address_list, $domain_list) = 
     ($CFG->params('GNS_TYPE'),
      $CFG->params('GNS_CREDENTIALS'),
      $CFG->params('GNS_ADDR_LIST'),
      $CFG->params('GNS_DOMAIN_LIST'));


   if("shared" eq lc($CFG->params('GNS_TYPE')))
   {
     if ($credentials ne undef) 
     {
       #Shared GNS is to be added.
       my $status = srvctl($run_as_owner,"add gns -clientdata $credentials");

       if (TRUE == ${status}) {
         trace ("add gns -clientdata $credentials ... passed");
       } else {
         print_error(180, "srvctl add gns -clientdata $credentials", $status);
         return FALSE;
       }
     }
     else
     {
       trace("GNS credentials file not provided");
       print_error(300,'GNS_CREDENTIALS');
       return FALSE;
     }
   } 
   elsif($domain_list ne undef){
      #A domain is provided.
      my $status =  &$srvctl_func($run_as_owner,
                          "add gns -i ${address_list} -d ${domain_list}");

      if (TRUE == ${status}) {
         trace ("add gns -i $address_list -d $domain_list ... passed");
      } else {
         print_error(180, "srvctl add gns -i $address_list -d $domain_list", $status);
         return FALSE;
      }
   } else {
      #No domain is provided.
      my $status =  &$srvctl_func($run_as_owner,
                          "add gns -i $address_list");

      if (TRUE == ${status}) {
         trace ("add gns -i $address_list ... passed");
      } else {
         print_error(180, "srvctl add gns -i $address_list", $status);
         return FALSE;
      } 
   }

   return TRUE;
}

sub add_scan 
{
   my $scanName = $CFG->params('SCAN_NAME');

   if ("shared" eq lc($CFG->params('GNS_TYPE')))
   {
     my $srvctl = crs_exec_path('srvctl');
     my @cmd    = ($srvctl, 'config', 'gns', '-S', '1');
     my @output = system_cmd_capture(@cmd);
     my $rc     = shift(@output);

     if (0 != $rc)
     {
       die(dieformat(180, "srvctl config gns -S 1", $rc));
     }
     else
     {
       my @line = grep(/gns_subdomain=/, @output);
       if (scalar(@line) > 0)
       {
         trace("   @line");
         my $result = (split(/:\s+/, $line[0]))[1];
         trace("result: $result");

         $result =~ s/{//g;
         $result =~ s/}//g;
         
         my $domain = (split(/=/, $result))[1];
         trace("Client GNS sub-domain: $domain");
         $scanName = $scanName."\.".$domain;
         trace("SCAN name with sub-domain suffix: $scanName");
       }
     }
   }

   my $srvctl_func = \&srvctl;
   if ((($^O eq "linux") || ($^O eq "solaris")) && ($CFG->AUTO))
   {
     trace("Call srvctl_tty");
     $srvctl_func = \&srvctl_tty; 
   }

   my $run_as_owner = FALSE;
   my $status = &$srvctl_func($run_as_owner, "add scan -n $scanName");

   if (${status}) {
      trace ("add scan=" . $scanName . " ... success");
   } else {
      print_error(180, "srvctl add scan -n $scanName", ${status});
      return FALSE;
   }

   return TRUE;
}

sub add_scan_listener 
{
   my $run_as_owner = TRUE;
   my $status = srvctl($run_as_owner, "add scan_listener -p " . 
                       $CFG->params('SCAN_PORT') . " -s");

   if (${status}) {
      trace ("add scan listener ... success");
   } else {
      print_error(180, "srvctl add scan_listener -p " . 
                  $CFG->params('SCAN_PORT') . " -s", ${status});
      return FALSE;
  }

  return TRUE;
}

sub add_mgmt_db_listener 
#-------------------------------------------------------------------------------
## Function: Add Management DB listener
## Args    : none
##------------------------------------------------------------------------------ 
{
   my @output;
   my $srvctl = crs_exec_path('srvctl');
   my $status = run_as_user2($CFG->params('ORACLE_OWNER'), \@output,
                                    ($srvctl, 'add', 'mgmtlsnr'));
   my @cmdout = grep(/(^PRCN-3004)/, @output);

   if ($status == 0 || scalar(@cmdout) > 0) {
        trace ("add mgmt db listener ... success");
   } else {
        print_error(180, "srvctl add mgmtlsnr", ${status});
        return FALSE;
   }
   return TRUE;
}

sub add_CVU
#------------------------------------------------------------------------------
# Function:  Create the cvu resource
# Args    :  checkInterval
#-------------------------------------------------------------------------------
{
   my $checkInterval = $_[0];
   my $run_as_owner = TRUE;
   my $cmdArgs = "add cvu";

   if ($checkInterval && $checkInterval ne "" && $checkInterval ne "0") {
      $cmdArgs = "add cvu -t $checkInterval";
   }

   trace ("running srvctl $cmdArgs");
   my $status = srvctl($run_as_owner, $cmdArgs);

   if (${status}) {
      trace ("add cvu ... success");
   } else {
      print_error(180, "srvctl add cvu", ${status});
      return FALSE;
   }

   return TRUE;
}

sub enable_CVU
#------------------------------------------------------------------------------
# Function:  enable the cvu resource
# Args    :  none
#-------------------------------------------------------------------------------
{
   my $run_as_owner = TRUE;
   trace ("running enable cvu");
   my $status = srvctl($run_as_owner, "enable cvu");

   if ($status) {
      trace ("enable cvu ... success");
   } else {
      print_error(180, "srvctl enable cvu", 1);
      return FALSE;
   }

   return TRUE;
}

sub disable_CVU
#------------------------------------------------------------------------------
# Function:  disable the cvu resource
# Args    :  none
#-------------------------------------------------------------------------------
{
   my $run_as_owner = TRUE;
   trace ("running disable cvu");
   my $status = srvctl($run_as_owner, "disable cvu");

   if ($status) {
      trace ("disable cvu ... success");
   } else {
      print_error(180, "srvctl disable cvu", 1);
      return FALSE;
   }

   return TRUE;
}

sub remove_CVU
#------------------------------------------------------------------------------
# Function:  remove the cvu resource and type
# Args    :  none
#-------------------------------------------------------------------------------
{
   my $run_as_owner = TRUE;
   my $status = srvctl($run_as_owner, "remove cvu -f");

   if (${status}) {
      trace ("remove cvu -f ... success");
   } else {
      print_error(180, "srvctl remove cvu -f", 1);
      return FALSE;
   }

   trace("remove cvu resource type");
   my $CRSCTL = crs_exec_path('crsctl');
   my @out = system_cmd_capture($CRSCTL, "delete", "type", "ora.cvu.type");
   $status = shift @out;
   if ($status != 0) {
     trace("remove cvu resource type failed @out");
     print_error(180, "$CRSCTL delete type ora.cvu.type", ${status});
     return FALSE;
   }

   return TRUE;
}

sub start_Nodeapps
#-------------------------------------------------------------------------------
# Function: Start nodeapps for static IP & DHCP
# Args    : [0] - DHCP_flag - TRUE if it's DHCP
#           [1] - nodes_to_start - list of nodes to be started
# Returns : TRUE  if success
#           FALSE if failed
#-------------------------------------------------------------------------------
{
   my $isDHCP             = shift;
   my $nodes_to_start_ref = shift;
   trace ("starting nodeapps...");
   trace ("DHCP_flag=$isDHCP");
   trace ("nodes_to_start=@$nodes_to_start_ref");
 
   if (0 == scalar(@$nodes_to_start_ref))
   {
     trace("nodes_to_start is null");
     return FALSE;
   }

   my $srvctl  = crs_exec_path('srvctl');
   my $success = TRUE;
   my $exit_value;
   my @output;
   my $cmd;
   my $rc;
   my $status;

   if (isBigCluster() && isRimNode())
   {
     trace("Cannot start vip on rim node");
     return TRUE;
   }
   
   foreach my $node (@$nodes_to_start_ref) {
      if (($isDHCP) && (! isFirstNodeToStart())) {
         # wait for vip resource to exist before start vip
         my $vip_exists = waitForVipRes();
         if (! $vip_exists) {
            print_error(10);
            return FALSE;
         }

         $cmd = "$srvctl start vip -i $node";
      } 
      else {
         $cmd = "$srvctl start nodeapps -n $node";
      }

      $rc = `$cmd`;
      $exit_value=$?>>8;

      trace("exit value of start nodeapps/vip is $exit_value");
      if ( $exit_value != 0) {
         my @lines = split("\n",$rc);
         trace("output for start nodeapps is  @lines");
         @output=grep(!/(^PRKO-2419|^PRKO-242[0-3])/,@lines);
         trace("output of startnodeapp after removing already started mesgs is @output");

         if (scalar(@output) >= 1) {
            if (($CFG->UPGRADE) && (scalar(grep(/CRS-2546/, @output)) > 0))
            {
              # Start nodeapp is invoked on the last node during uprade from pre-11.2
              trace("Proceed with the upgrade if the target node is not online at this time");
            }
            else
            {
              print_error(180, $cmd, $exit_value);
              $success = FALSE;
            }
         } else {
            trace ("$cmd ... passed");
         }
      }
   }

   return $success;

}

sub start_GNS
#---------------------------------------------------------------------
# Function: Start GNS
# Args    : none
# Returns : TRUE  if success
#           FALSE if failed
#---------------------------------------------------------------------
{
   if ($CFG->params('GNS_CONF') ne "true") {
      trace ("GNS is not to be configured - skipping");
      return TRUE;
   }

   # check if gns is running
   my $srvctl    = crs_exec_path('srvctl');
   my $save_lang = $ENV{LANG};
   $ENV{LANG}    = "";

   open OPUT, "$srvctl status gns|";
   my @output = grep(/not running/, <OPUT>);
   close OPUT;
   $ENV{LANG} = $save_lang;

   if (scalar(@output) > 0) {
      trace ("gns is not running");
   }
   else {
      trace ("gns is running");
      return TRUE;
   }

   # start gns
   my $run_as_owner = FALSE;
   my $status = srvctl($run_as_owner, "start gns");

   if (${status}) {
      trace ("start gns ... passed");
   } else {
      print_error(107);
      return FALSE;
   }

   return TRUE;
}

sub start_scan 
{
   my $run_as_owner = FALSE;
   my $status = srvctl($run_as_owner, "start scan");

   if (${status}) {
      trace ("start scan ... success");
   } else {
      print_error(110);
      return FALSE;
   }

   return TRUE;
}

sub start_scan_listener 
{
   my $run_as_owner = TRUE;
   my $status = srvctl($run_as_owner, "start scan_listener");

   if (${status}) {
      trace ("start scan listener ... success");
   } else {
      print_error(111);
      return FALSE;
   }

   return TRUE;
}

=head2 oifcfgiflst_to_instlststr

 Create GPnP networks list based on oifcfg info
 Example of output:
 "Local Area Connection 3"/140.87.128.0:public,"Local Area Connection 4"/140.87.136.0:cluster_interconnect|public
 or 
 eth0/10.0.100.0:cluster_interconnect,eth1/140.87.4.0.0:public
 Adaptor name can be quoted. /\<>|"*? are not legal for in adapter name (note,
 spaces or commas can appear). List is space-separated.
 For the sake of "oifcfg iflist" compatibility, UNKNOWN/PRIVATE/LOCAL types
 recognized (UNKNOWN mapped to public, PRIVATE mapped to cluster_interconnect,
 and LOCAL skipped. oifcfg types can be combined (comma-separated) - inst
 list uses | separator, though installer never produces interfaces with
 multiple types.

=head3 Parameters

  Reference to array of oifcfg-style output, e.g.
   ("eth0  10.0.0.0  PRIVATE", "eth1  140.87.4.0  UNKNOWN") 
   ("Local Area Connection 3 140.87.128.0 PUBLIC", i
    "Local Area Connection 4 140.87.136.0 CLUSTER_INTERCONNECT,PUBLIC" )
   
=head3 Returns 

  returns a string with installer-style net info.

=cut

sub oifcfgiflst_to_instlststr
{
   my $intfsref  = $_[0]; # ref
   my @intfs     = @{$intfsref};
   my $s_instiflist = '';
   foreach (0..$#intfs) {
      my $idef = $intfs[$_];
      my @intf = oifcfg_intf_parse( $idef );

      $s_instiflist = oifcfg_intf_to_instlststr( \@intf, $s_instiflist );
   }
   trace ("inst netlst:\"".$s_instiflist."\""); 
   return $s_instiflist;
}

=head2 oifcfg_intf_parse

  Parse a single net interface description produced by oifcfg cmd.
  For example:
  a) oifcfg iflist output:
   "eth0  10.0.0.0  PRIVATE  255.255.252.0",
   "eth1  140.87.4.0  UNKNOWN  255.255.252.0",
   "Local Area Connection 3  140.87.128.0  PRIVATE"
  b) oifcfg getif output:
   "Local Area Connection 4  140.87.136.0  global  cluster_interconnect,public"

=head3 Parameters

  A string containing oifcfg interface definition, see examples above.
  (Other strings, such as warnings, etc., must be filtered out.)

=head3 Returns

 @returns an array of interface parameters:
   ($adapter_name, $network, $node_name, $type_list, $mask)
   Where:
   adapter_name is the name of net adapter, unquoted;
   network is a network bits of adapter address;
   node_name is 'global' for cluster-wide config, or node name if node-specific;
   type_list is a comma-separated list of network types (valid values are
             unknown|local|public|private|cluster_interconnect);
   mask is a mask bits;
   If any of the parameters was not defined, undef returned in its place.

=cut

sub oifcfg_intf_parse
{
   my $idef = $_[0];
   my $valid_types = "(local|public|private|unknown|cluster_interconnect|asm)([,](local|public|private|unkown|cluster_interconnect|asm))*";
   my $an_;
   my $ada;
   my $net;
   my $nod;
   my $typ;
   my $msk;
   my $iaddr;
   my $n;
   trace ("intf: $idef");

   $idef =~ s/^\s+|\s+$//g;
   $n   = rindex( $idef, ' ' );
   $an_ = substr( $idef, 0, $n );
   $an_ =~ s/\s+$//;

   $typ = substr( $idef, $n+1 );
   if ($typ !~ m/$valid_types/i) { # if cannot be type, check if mask
      $msk = $typ;
      $typ = undef;

      # validate mask
      $iaddr = ipv4_atol($msk); # toberevised: +ipv6 - inet_pton
      if (! defined $iaddr) {
         $msk = undef;
      } else {
         $msk = ipv4_ltoa($iaddr); # toberevised: +ipv6 - inet_ntop
      }
      $iaddr = undef;
      if (defined $msk)
      {
         $n   = rindex( $an_, ' ' );
         $typ = substr( $an_, $n+1 );
         $an_ = substr( $an_, 0, $n );
         $an_ =~ s/\s+$//;
      }
   }
   if ($typ !~ m/$valid_types/i) {
      $typ = undef;
   }
   if (defined $typ) {
      $n = rindex( $an_, ' ' );
      if (1 <= $n) {
         $ada = substr( $an_, 0, $n );
         $ada =~ s/\s+$//;
         $net = substr( $an_, ($n+1) );

         # validate address, if not addr, must be scope (nodename/global)
         $iaddr = ipv4_atol($net); # toberevised: +ipv6 - inet_pton
         if ((! defined $iaddr) || ($iaddr == 0)) {
           $nod = $net;
           $net = undef;
           $n = rindex( $ada, ' ' );
           if (1 <= $n) {
             $net = substr( $ada, ($n+1) );
             $ada = substr( $ada, 0, $n );
             $ada =~ s/\s+$//;
           }
           # validate address
           $iaddr = ipv4_atol($net); #toberevised: +ipv6 -inet_pton
         }
         if ((! defined $iaddr) || ($iaddr == 0)) {
           $net = undef;
         } else {
           $net = ipv4_ltoa($iaddr); # toberevised: +ipv6 - inet_ntop
         }
         $iaddr = undef;
      }
   }
   trace ("intf parsed: -$ada-$net-$nod-$typ-$msk-=");

   return ($ada, $net, $nod, lc($typ), $msk );
}

=head2 ipv4_atol
  Convert a string with decimal dotted ipv4 to network-ordered long
  Note: this is quite similar to inet_aton(); however, it does not tries
  to resolve hostnames as inet_aton does.

=head3 Parameters

  String containing decimal-dotted ipv4 address value.

=head3 Returns

 @returns a network-ordered long ipv4 address value.

=cut

sub ipv4_atol
{
  return unpack('N',pack('C4',split(/\./,shift)));
}

=head2 ipv4_ltoa
  Convert a network-ordered long ipv4 address value to a
  string decimal dotted ipv4 notation.

=head3 Parameters

  Long containing network-ordered ipv4 address value.

=head3 Returns

 @returns a string dotted-decimal ipv4 address value.

=cut

sub ipv4_ltoa
{
  return inet_ntoa(pack('N',shift));
}

sub oifcfg_intf_to_instlststr
{
   my $idefref  = $_[0]; # ref # ref to parsed interface definition
   my $s_instiflist = $_[1];   # list of NETWORKS, installer-style

   my $ada;
   my $net;
   my $nod;
   my $typ;
   my $msk;
   ($ada, $net, $nod, $typ, $msk ) = @{$idefref};
   $s_instiflist = '' if (! defined $s_instiflist);

   if ((! defined $typ) || (! defined $net) || (! defined $ada)) {
      return; # bad intf definition
   }
   # For the sake of "oifcfg iflist" compatibility, UNKNOWN/PRIVATE/LOCAL types
   # recognized (UNKNOWN mapped to public,
   # PRIVATE mapped to cluster_interconnect,
   # and LOCAL skipped. oifcfg types can be combined (comma-separated) - inst
   # list uses | separator, though installer never produces interfaces with
   # multiple types.
   #
   if ($typ =~ m/LOCAL/i) {
      return; # skip "do_not_use" itf
   }
   $typ =~ s/(UNKNOWN|unknown)/public/g;
   $typ =~ s/(PRIVATE|private)/cluster_interconnect/g;
   $typ =~ s/,/|/g;  # replace separator

   # tabs, \/|"'*:<>? are normally illegal in adapter names
   if ($ada =~ /[ ,:<>\t\^\(\)\\\/\*\?\|\[\]\+]/) {
      $ada = '"'.$ada.'"';
   }
   if (!($s_instiflist eq '')) {
      $s_instiflist .= ',';
   }
   $s_instiflist .= $ada.'/'.$net.':'.lc($typ);
   trace ("inst netlst:\"".$s_instiflist."\"");

   return $s_instiflist;
}

sub isResRunning
#-------------------------------------------------------------------------------
# Function:  Check if resource is running
# Args    :  0 - resource
#         :  1 - node name
# Returns :  TRUE  if     running
#            FALSE if not running
#-------------------------------------------------------------------------------
{
   my $res      = $_[0];
   my $nodename = $_[1];
   my $node     = ($nodename) ? ($nodename) : ($CFG->HOST);  
   my $crsctl   = crs_exec_path("crsctl");

   trace("Check if resource '$res' is ONLINE on node '$node'");
   my @cmd      = ($crsctl, 'status', 'resource', $res, '-n', $node);
   my @out      = system_cmd_capture(@cmd);
   my $rc       = shift @out;

   if ($rc == 0)
   {
      my @reStat = grep(/STATE=ONLINE/, @out);
      if (scalar(@reStat) > 0) {
         trace("$res is online");
         return TRUE;
      }
      else {
         trace("$res is not online");
         return FALSE;
      }
   }
    
   return FALSE;
}

sub isPrereqIgnored
{
  my $isprereqignored = FALSE;
  if ($CFG->params('USER_IGNORED_PREREQ') =~ m/true/i) {
      $isprereqignored = TRUE;
   }
  trace("USER_IGNORED_PREREQ  is set to $isprereqignored");
  return $isprereqignored;
}


# delete the bdb files in bdbloc.
sub crf_delete_bdb
{
  my $bdbloc = $_[0];
  my $cwd = cwd();

  if ($bdbloc ne "" and -d $bdbloc)
  {
    # remove files which we created.
    chdir $bdbloc or print_error(355, $bdbloc);
    opendir(DIR, "$bdbloc") || print_error(356, $bdbloc);
    trace("Removing contents of BDB Directory $bdbloc\n");
    my $crfbdbfile;
    my @ldbfiles = grep(/\.ldb$/,readdir(DIR));
    foreach $crfbdbfile (@ldbfiles)
    {
      unlink ($crfbdbfile);
    }

    # database files
    rewinddir (DIR);
    my @bdbfiles = grep(/\.bdb$/,readdir(DIR));
    foreach $crfbdbfile (@bdbfiles)
    {
      unlink ($crfbdbfile);
    }

    # env files
    rewinddir (DIR);
    my @dbfiles = grep(/__db.*$/,readdir(DIR));
    foreach $crfbdbfile (@dbfiles)
    {
      unlink ($crfbdbfile);
    }

    # archive log
    rewinddir (DIR);
    my @bdblogfiles = grep(/log.*$/,readdir(DIR));
    foreach $crfbdbfile (@bdblogfiles)
    {
      unlink ($crfbdbfile);
    }
    closedir (DIR);
    # change back to current working directory.
    chdir $cwd or print_error(355, $cwd);
    trace("Changed to older working directory $cwd\n");
  }
}

sub unlockHAHomefordeinstall
{
   trace ("Unlock Oracle Restart home...");
   my $exclfile = $_[0];
   my $unlock_hahome;
   my $status = SUCCESS;

   trace("Exclude file used is $exclfile");

   if (!$CFG->UNLOCK) { $CFG->UNLOCK(TRUE); }

   #Try to get the home path from olr.loc
   $unlock_hahome = s_get_olr_file ("crs_home");
   trace ("Home location in olr.loc is $unlock_hahome");

   if (! $unlock_hahome)
   {
      $unlock_hahome = $CFG->hahome;
   }

   if ($CFG->hahome)
   {
     if (($CFG->NOCRSSTOP) && ($unlock_hahome eq $CFG->hahome))
     {
       trace("Attempting to unlock an active Oracle HA home");
     }
     elsif ($CFG->NOCRSSTOP) 
     {
       $unlock_hahome = $CFG->hahome;
     }         
   }
   
   trace("Oracle HA home to unlock is $unlock_hahome");

   # validate if crshome exists
   if (! -e $unlock_hahome) {
      print_error(46, $unlock_hahome);
      trace ("Oracle Restart home: $unlock_hahome not found\n");
      return FALSE;
   }

   if (! $CFG->NOCRSSTOP)
   {
     $status = stopOracleRestart(); 
   }

   # check the status of HA stack
   if ($status) {
      s_reset_crshome1 ($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'),
                        755, $unlock_hahome,$exclfile);
      print_error(347, $unlock_hahome);
   }
   else { 
      print_error(348);
   }
}

sub stopOracleRestart
#-------------------------------------------------------------------------------
# Function: Stop Oracle Restart Stack
# Returns : TRUE  if success
#           FALSE if failed
#-------------------------------------------------------------------------------
{
   my $res      = $_[0];
   my $success  = TRUE;
   my $crsctl;

   if ($CFG->platform_family eq "windows") {
      $crsctl = crs_exec_path("crsctl.exe");
   } else {
      $crsctl = crs_exec_path("crsctl");
   }

   if (! -e $crsctl) {
      print_error(13, $crsctl);
      trace ("$crsctl does not exist to proceed stop clusterware");
      return FALSE;
   }

   # stop resource
  
   my  @cmd = ($crsctl, 'stop', 'has', '-f');
   trace("Executing @cmd");
   my @out       = system_cmd_capture(@cmd);
   my $rc        = shift @out;

   # Allow HA daemons to shutdown in 10sec
   sleep 10;

   # check if ohasd is  still up
   if (! checkServiceDown("ohasd")) {
         print_error(192);
         $success = FALSE;
   }

   return $success;
}

sub unlockCRSHomefordeinstall
{
   trace ("Unlock crshome...");
   my $exclfile = $_[0];
   my $statusCRS;
   
   trace ("crshome passed is " . $CFG->UNLOCK_HOME);
   trace("Exclude file used is $exclfile");
   my $unlock_crshome1;

   if (!$CFG->UNLOCK) { $CFG->UNLOCK(TRUE); }

   #Try to get the home path from olr.loc
   $unlock_crshome1 = s_get_olr_file ("crs_home");
   trace ("Home location in olr.loc is $unlock_crshome1");

   if (! $unlock_crshome1) {
      $unlock_crshome1 = $CFG->UNLOCK_HOME;
   }

   #NOCRSSTOP option to be passed to unlock an inactive grid home
   if ($CFG->UNLOCK_HOME) {
        if ($CFG->NOCRSSTOP && ($CFG->UNLOCK_HOME eq $unlock_crshome1)) {
           error("You can't unlock an active CRS home");
        }
        elsif ($CFG->NOCRSSTOP) {
           $unlock_crshome1 = $CFG->UNLOCK_HOME;
        }
   }

   trace ("unlock crs home is $unlock_crshome1");

   # validate if crshome exists
   if (! -e $unlock_crshome1) {
      print_error(46, $unlock_crshome1);
      trace ("crshome: $unlock_crshome1 not found\n");
      return FALSE;
   }

   if (! $CFG->NOCRSSTOP) {
      $statusCRS = stopClusterware($unlock_crshome1, "crs");
   }
   else {
      $statusCRS = SUCCESS;
   }

   if ($statusCRS) {
      unlockHome($CFG->params('ORACLE_OWNER'),
                 $CFG->params('ORA_DBA_GROUP'), 755, $unlock_crshome1);
      print_error(347, $unlock_crshome1);
   }
   else {
      print_error(349);
   }
}


sub unlockHome
#-------------------------------------------------------------------------------
# Function:  unlocks the GI Home for patching. sets the owner of all the
#            files/dirs in crsconfig_fileperms and crsconfig_dirs to GI Home
#            owner. This is sufficient for patching GI Home.
# Args    :  none
# Returns :  none
#-------------------------------------------------------------------------------
{
   my ($owner, $group, $perms, $basedir) = @_;
   my %dirown;
   my %fileown;
   my ($a, $dirname, $own, $grp, $perm, $fname);
   my $ORA_CRS_HOME;

   trace("The GI home to unlock: $basedir");
   $ORA_CRS_HOME = ($basedir) ? $basedir :($CFG->ORA_CRS_HOME);

   trace("Unlocking the GI home: $ORA_CRS_HOME");

   my $platform = s_get_platform_family();
   if ($platform eq 'windows')
   {
     $ORA_CRS_HOME =~ s/\\/\\\\/g;
   }

   my $SUPERUSER    = $CFG->SUPERUSER;
   my $wrapdir_crs  = catdir($ORA_CRS_HOME, "crs", "utl");
   my @dirs         = read_file (catfile($wrapdir_crs, "crsconfig_dirs"));
   my @fileperms    = read_file (catfile($wrapdir_crs, "crsconfig_fileperms"));

   foreach my $line (@dirs) {
      chomp ($line);
      next if ($line =~ /^#|\$|^\s*$/);

      ($a, $dirname, $own, $grp, $perm) = split(/ /, $line);
      if ($own =~ $SUPERUSER) {
         $dirown{$dirname} = $own;
      }
      else {
         next;
      }
   }

   foreach my $line (@fileperms) {
      chomp ($line);
      next if ($line =~ /^#|\$|^\s*$/);
      ($a, $fname, $own, $grp, $perm) = split(/ /, $line);
      if ($own =~ $SUPERUSER) {
         $fileown{$fname} = $own;
      }
      else {
         next;
      }
   }

   foreach my $dir (keys%dirown) {
      next if ($dir !~ m/^$ORA_CRS_HOME.*/);
      s_set_ownergroup ($owner, $group, $dir);
   }

   foreach my $file (keys%fileown) {
      next if ($file !~ m/^$ORA_CRS_HOME.*/);
      s_set_ownergroup ($owner, $group, $file);
   }
}


sub lockAcfsFiles
{
   my %fileown;
   my ($a, $own, $grp, $perm, $fname);
   my $ORA_CRS_HOME = $CFG->ORA_CRS_HOME;
   my $wrapdir_crs  = catfile ($ORA_CRS_HOME, "crs", "utl");
   my @fileperms    = read_file (catfile($wrapdir_crs, "crsconfig_fileperms"));
   
   foreach my $line (@fileperms) {
      chomp ($line);
      next if ($line =~ /^#|\$|^\s*$/       || 
               $line !~ m/^.*acfs.*/        ||
               $line =~ m/^.*install.*/     );
      ($a, $fname, $own, $grp, $perm) = split(/ /, $line);
      if ($own =~ $CFG->HAS_USER) {
         $fileown{$fname} = $own;
      }
      else {
         next;
      }
   }     
            
   foreach my $file (keys%fileown) {
      next if ($file !~ m/^$ORA_CRS_HOME.*/); 
      s_set_ownergroup ($CFG->SUPERUSER,  $CFG->params('ORA_DBA_GROUP'), 
                        $file);  
   }
}


sub modifyparamfile
{
   my $param = $_[0];
   my $paramval = $_[1];

   trace("modify $param $CFG->paramfile");

   my $params_file = $CFG->paramfile;
   my @params_table = read_file ($params_file);
   my $updateParamsFile = FALSE;
   my $ix = 0;

   foreach my $rec (@params_table) {
      chomp($rec);
      if ($rec =~ m/^$param=/) {
         my ($key, $value) = split (/=/, $rec);
         $params_table[$ix] = $param . "=" . "$paramval";
         $updateParamsFile = TRUE;
         last;
      }

      $ix++;
   }
   if ($updateParamsFile) {
      # save original params file
      my $save_file = catfile (dirname($params_file), 'crsconfig_params.saved');
      copy_file ($params_file, $save_file);

      # delete old file and create new file
      if (s_remove_file($params_file)) {
         open (SFILE, ">$params_file")
            || die ("Can't open $params_file to write: $!");

         foreach my $rec (@params_table) {
            chomp($rec);
            print SFILE "$rec\n";
         }

         close SFILE;
      }
   }
}

sub stopClusterware
#-------------------------------------------------------------------------------
# Function: Stop crs / cluster 
# Args    : oracle home, Resource (crs/cluster)
# Returns : TRUE  if success
#           FALSE if failed
#-------------------------------------------------------------------------------
{
   my ($home, $res) = @_;
   my $success      = TRUE;
   my $crsctl;

   if (! defined $home) {
      $crsctl = crs_exec_path('crsctl');
   }
   else {
     $crsctl = catfile( $home, 'bin', 'crsctl' );
   }

   # stop resource
   my @cmd;
   if ($res eq 'crs') {
      @cmd = ($crsctl, 'stop', 'crs', '-f');
   } 
   else {
      @cmd = ($crsctl, 'stop', 'resource', '-all', '-init');
   } 
 
   my @out = system_cmd_capture (@cmd);
   my $rc = shift @out;
   trace ("@cmd"); 

   if (! checkServiceDown("cluster")) {
      print_error(357);
      return FALSE;
   }

   # check if ohasd & crs are still up
   if ($res eq 'crs') {
      if (! checkServiceDown("ohasd")) {
         print_error(191); 
         $success = FALSE;
      }
   }

   return $success;
}

sub queryVoteDisks
{
#-------------------------------------------------------------------------------
# Function: Query Voting disks
# Args    : none
# Returns : List of voting disks/files
#-------------------------------------------------------------------------------
   # Check if CRS is up
   my $crsctl = crs_exec_path('crsctl');
   my $cluster_is_up = check_service ("cluster", 20);
   my $crs_is_up = check_service ("ohasd", 10);
   my $start_exclusive = FALSE;
   my @css_votedisk;

   if (!$crs_is_up) 
   {
     trace("OHASD is not up. So starting CRS exclusive");
     start_service("crsexcl");
     $crs_is_up = TRUE;
   }
   else 
   {
     trace("OHASD is already up.");

     if (!$cluster_is_up) 
     {
       trace("Starting CSS exclusive");
       $start_exclusive = TRUE;
       my $css_rc = CSS_start_exclusive();
       if ($css_rc != CSS_EXCL_SUCCESS) 
       {
         $start_exclusive = FALSE;
         trace ("CSS failed to enter exclusive mode to extract votedisk");
       }
     }
   }

   if (($crs_is_up) || ($start_exclusive)) 
   {
      trace("Querying CSS vote disks");
      open (QUERY_VOTEDISK, "$crsctl query css votedisk|");
      @css_votedisk = (<QUERY_VOTEDISK>);
      chomp @css_votedisk;
      close QUERY_VOTEDISK;

      if ($start_exclusive) 
      {
         CSS_stop();
      }
    }

   return @css_votedisk;
}

sub extractVotedisks
#-------------------------------------------------------------------------------
# Function: Extract Voting disks
# Args    : none
#-------------------------------------------------------------------------------
{
  my @votedisk_list;

  my @css_votedisk = queryVoteDisks();

  foreach my $votedisk (@css_votedisk) 
  {
    trace("Voting disk is : $votedisk");
    # get line contains ' (/'
    if ($votedisk =~ / \(/)
    {
      # $votedisk contains '1.  2 282bf2a833f54f02bf4befd002fa90d6
      # (/dev/raw/raw1) [OCRDG]'
      # parse $votedisk to get '/dev/raw/raw1'
      my $vdisk;
      my ($dummy, $text) = split (/\(/, $votedisk);
      ($vdisk, $dummy) = split (/\)/, $text);
      push (@votedisk_list, $vdisk);
    }
  }

  trace ("Vote disks found: @votedisk_list");
  return @votedisk_list;
}

####---------------------------------------------------------
#### Function for validating OCR config
# ARGS: 2
# ARG1 : ocrlocations
# ARG2 : isHas
sub validate_ocrconfig
{
   my $ocrlocations = $_[0];
   my $isHas        = $_[1];

   if (!$ocrlocations) {
      print_error(9);
      return FAILED;
   }

   trace ("OCR locations = " . $ocrlocations);

   # OSD to validate OCR config
   s_validate_ocrconfig ($ocrlocations, $isHas) or return FAILED;

   return SUCCESS;
}

sub add_ons
{
   my $run_as_owner = TRUE;
   my $status       = srvctl($run_as_owner, 'add ons');

   if ($status) {
      trace ("srvctl add ons ... success");
   }
   else {
      print_error(180, "srvctl add ons", $status);
      return FALSE;
   }

   return TRUE;
}

sub upgrade_local_OCR
{
   my $ocrconfig = catfile($CFG->ORA_CRS_HOME, "bin", "ocrconfig");
   my $ckptName  = "ROOTCRS_OLR";
   my $status;

   trace ("Creating or upgrading Oracle Local Registry (OLR)");
   if ($CFG->SIHA) {
      $status = run_as_user($CFG->params('ORACLE_OWNER'),
                            "$ocrconfig -local -upgrade");
   }
   else {
      my @cmd = ($ocrconfig, '-local', '-upgrade',
                 $CFG->params('ORACLE_OWNER'),
                 $CFG->params('ORA_DBA_GROUP'));
      $status = system_cmd(@cmd);
   }

   if (0 == $status) {
      trace ("OLR successfully created or upgraded");
   }
   else {
      trace("$ocrconfig -local -upgrade failed with error: $status");
      writeCkpt($ckptName, CKPTFAIL);
      die(dieformat(169));
   }

   return SUCCESS;
}

sub create_OLR_keys
{
   my $clscfg   = catfile($CFG->ORA_CRS_HOME, 'bin', 'clscfg');
   my $ckptName = "ROOTCRS_OLR";
   my $status;
   my @out = ();

   # create keys in OLR
   trace ("$clscfg -localadd");

   if ($CFG->SIHA) {
      $status = run_as_user($CFG->params('ORACLE_OWNER'), "$clscfg -localadd");
   }
   else {
      @out = system_cmd_capture("$clscfg -localadd");
      $status = shift @out;
   }

   if (0 == $status) {
      trace ("Keys created in the OLR successfully");
      writeCkpt($ckptName, CKPTSUC);
      $CFG->wipCkptName("ROOTCRS_STACK");
   }
   else {
      error ("Failed to create keys in the OLR, rc = $status, Message:\n ",
             "@out \n");
      writeCkpt($ckptName, CKPTFAIL);
      die(dieformat(188));
   }

   return SUCCESS;
}

sub checkOCR
#-------------------------------------------------------------------------------
# Function: checks if the OCR is configured properly and not corrupted.
# Args    : none
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
{
   my $crs_home = $_[0];
   my $cmd = catfile($crs_home, 'bin', 'ocrcheck');
   my @out = system_cmd_capture($cmd, "-debug");
   my $rc  = shift @out;
   my $check_passed = FAILED;

   trace("checkOCR rc=$rc");
   my @cmdout1 = grep(/(^PROT-708)/, @out);
   my @cmdout2 = grep(/(^PROT-715)/, @out);
   my @cmdout3 = grep(/(^PROT-721)/, @out);

   if ($rc == 0 && scalar(@cmdout1) == 0 &&
       scalar(@cmdout2) == 0 && scalar(@cmdout3) == 0)
   {
     $check_passed = SUCCESS;
     trace ("OCR check: passed");
   }
   else
   {
     trace ("OCR check: failed");
   }

   return $check_passed;
}

=head2 getNodeConfigRole

  Get the configured role for current node or a given node 

=head3 Parameters

  ARG1: Given node name; refers to the current node if not supplied
        
=head3 Returns

  Valid node role or undef if unsuccessful
  
=head3 Usage
  
  my $role = getNodeConfigRole();
  or
  my $role = getNodeConfigRole($node);
    
=cut

sub getNodeConfigRole
{ 
  my $node = $_[0];
  my $role = undef;
  my @cmd;
  
  my $nodeName = (defined $node) ? $node: "local";
  trace("Getting the configured node role for the $nodeName node");
  
  my $crsctl = crs_exec_path('crsctl');
  if (defined $node)
  { 
    @cmd = ($crsctl, 'get', 'node', 'role', 'config', '-node', $node);
  }
  else
  { 
    @cmd = ($crsctl, 'get', 'node', 'role', 'config');
  }

  my @output = system_cmd_capture(@cmd);
  my $rc = shift @output;
  if (0 != $rc)
  {
    if (defined $node)
    {
      print_error(376, $node);
    }
    else
    {
      print_error(378);
    }
  }
  else
  {
    if (scalar(grep(/hub/, @output)) > 0)
    {
      $role = NODE_ROLE_HUB;
    }
    elsif (scalar(grep(/leaf/, @output)) > 0)
    {
      $role = NODE_ROLE_RIM;
    }
    elsif (scalar(grep(/auto/, @output)) > 0)
    {
      $role = NODE_ROLE_AUTO;
    }
    else
    {
      trace("Invalid node role: @output");
    }
  }

  if (defined $role)
  {
    trace("The configured node role for the $nodeName node is $role");
  }

  return $role;
}

=head2 isHubNode

  Check if the current node is a hub node

=head3 Parameters

  None 

=head3 Returns
  
  TRUE or FALSE
  
=head3 Usage
  
=cut

sub isHubNode
{
  if (NODE_ROLE_HUB eq getNodeConfigRole())
  {
    trace("the node role is hub");
    return TRUE;
  }

  return FALSE;
}

sub isRimNode
{
  if (NODE_ROLE_RIM eq getNodeConfigRole())
  {
    trace("the node role is leaf");
    return TRUE;
  }

  return FALSE;
}

=head2 isAutoNode

  Check if the current node is a auto node

=head3 Parameters

  None 

=head3 Returns
  
  TRUE or FALSE
  
=head3 Usage
  
=cut

sub isAutoNode
{
  if (NODE_ROLE_AUTO eq getNodeConfigRole())
  {
    trace("the node role is auto");
    return TRUE;
  }

  return FALSE;
}

=head2 setCurrentNodeRole

  Set the role for the local node

=head3 Parameters

  Args: None

=head3 Returns

  SUCCESS or FAILED
  
=head3 Usage
  
  my $ret = setCurrentNodeRole();
    
=cut
    
sub setCurrentNodeRole
{
  my $role;
  my $status = SUCCESS;
  my $localNode = $CFG->HOST;
  
  if (!isBigCluster())
  {
    # Return at once if not big cluster
    trace("No need to set the role for the local node in a regular cluster");
    return SUCCESS;
  } 
  
  trace("Setting the role for the local node '$localNode'");

  if (isAddNode($localNode, $CFG->params('NODE_NAME_LIST')))
  {
    trace("New node $localNode to be added");
    my @new_hosts = split (/,/, $CFG->params('CLUSTER_NEW_HOST_NAMES'));
    my @new_roles = split (/,/, $CFG->params('CLUSTER_NEW_NODE_ROLES'));
    my $nbr_of_hosts = scalar(@new_hosts);
    my $nbr_of_roles = scalar(@new_roles);
    my $ix = 0;

    if ($nbr_of_hosts != $nbr_of_roles)
    {
      die(dieformat(438, $CFG->ORA_CRS_HOME));
    }

    foreach my $host (@new_hosts)
    {
      chomp $host;
      if ($localNode eq lc($host))
      {
        last;
      }
      $ix++;
    }

    $role = $new_roles[$ix];
    trace("The new node's role is: $role");
    $role = lc($role);
  }

  if (!$role)
  {
    my $hubNodes = $CFG->params('HUB_NODE_LIST');
    if ($hubNodes =~ /\b$localNode\b/i)
    {
      $role = NODE_ROLE_HUB;
    }
  }

  if (!$role)
  {
    my $rimNodes = $CFG->params('RIM_NODE_LIST');
    if ($rimNodes =~ /\b$localNode\b/i)
    {
      $role = NODE_ROLE_RIM;
    }
  }

  if ($role)
  {
    trace("The current node's role is: $role");
    if ($role ne NODE_ROLE_HUB)
    {
      $status = setNodeRole($role);
    }
  }
  else
  {
    trace("The current node '$localNode' is not classified");
    $status = FAILED;
  }

  return $status;
}

sub setNodeRole
{
   my $role = $_[0];
   my $status = SUCCESS;

   my $crsctl = crs_exec_path('crsctl');
   my @cmd = ($crsctl, 'set', 'node', 'role', $role);
   my @output = system_cmd_capture(@cmd);
   my $rc = shift @output;
   if (0 != $rc)
   {
      print_error(368, $role);
      $status = FAILED;
    }
    return $status;
}


=head2 setHubSize

  Set the maximum size of hub nodes 

=head3 Parameters

  None

=head3 Returns

  SUCCESS or FAILED
  
=head3 Usage
    
=cut

sub setHubSize
{
  my $hubSize = $CFG->params('HUB_SIZE');
  trace("Setting the hub size to $hubSize");

  my $crsctl = crs_exec_path('crsctl');
  my @cmd = ($crsctl, 'set', 'cluster', 'hubsize', $hubSize);
  my @output = system_cmd_capture(@cmd);
  my $status = shift @output;
  if (0 != $status)
  {
    print_error(369, $hubSize);
    return FAILED;
  }

  return SUCCESS;
}

=head2 isBigCluster

  Check if it's a big cluster environment

=head3 Parameters

  None 

=head3 Returns

  TRUE : A big cluster
  FALSE: Not a big cluster
  
=head3 Usage
  
  my $bc = isBigCluster();
   
=cut

sub isBigCluster
{
  if ("true" eq lc($CFG->params('BIG_CLUSTER')))
  {
    trace("Set BC to 1");
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}

=head2 createCredDomain

  The subroutine for creating credential domain in OCR/OLR

=head3 Parameters

  Args: Domain name

=head3 Returns

  SUCCESS or FAILED
  
=head3 Usage
  
   my $ret = createCredDomain('ASM', 'OCR'); 
=cut

sub createCredDomain
{
  my $domain = $_[0];
  my $location = $_[1];

  my @out;
  my @program;
  my $rc;
  my $crsctl = crs_exec_path('crsctl');

  my $local;
  if ("olr" eq lc($location))
  {
    $local = "-local";
  }
  @program = ($crsctl, 'add', 'credmaint', '-path', $domain, $local);
  @out = system_cmd_capture(@program);
  $rc  = shift @out;
  if (0 != $rc &&
      scalar(grep(/10405/, @out)) == 0)
  {
    trace("Failed to add $domain credential domain, error: @out");
    return FAILED;
  }
  
  @program = ($crsctl, 'setperm', 'credmaint',
              '-o',  $CFG->params('ORACLE_OWNER'), '-path', $domain, $local);
  @out = system_cmd_capture(@program);
  $rc  = shift @out;
  if (0 != $rc)
  {
    trace("Failed to set perms on $domain credential domain, error: @out");
    return FAILED;
  }

  return SUCCESS;
}

#-------------------------------------------------------------------------------
# Function: Enables flex asm on non-first nodes by importing the credentials created on the
# first node.
# Args    : none
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
sub importASMCredentials
{
  my @output;
  my $rc = -1;

  my $KFOD = catfile($CFG->ORA_CRS_HOME, 'bin', 'kfod');
  my $wrap = 'wrap=' . get_asm_cred_file();
  my @program = ($KFOD, 'op=credimport', $wrap, 'olr=TRUE', 'force=TRUE');

  $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@output, @program);
  trace("kfod op=credimport rc: $rc");
  if (0 != $rc)
  {
    trace("Failed to enable flex ASM on local node, error: @output");
    return FAILED;
  }
  else
  {
    trace("Enabled flex ASM on local node");
  }

  return SUCCESS;
}

#-------------------------------------------------------------------------------
# Function: Enables flex asm by creating credentials on the first node. They are
# propagated to the other nodes.
# Args    : none
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
sub createASMCredentials
{
  my @output;
  my $rc = -1;

  my $KFOD = catfile($CFG->ORA_CRS_HOME, 'bin', 'kfod');
  my $wrap = 'wrap=' . get_asm_cred_file();
  my @program = ($KFOD, 'op=credremote', $wrap);

  $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@output, @program);
  trace("kfod op=credremote rc: $rc");
  if (0 != $rc)
  {
    trace("Failed to create credentials for flex ASM, error: @output");
    return FAILED;
  }
  else
  {
    trace("Created credentials for flex ASM");
  }

  return SUCCESS;
}

#-------------------------------------------------------------------------------
# Function: Removes the remote asm setting in the gpnp profile. This is called during 
# deconfig.
# Args    : none
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
sub disableRemoteASM
{
  my @output;
  my $rc = -1;

  my $KFOD = catfile($CFG->ORA_CRS_HOME, 'bin', 'kfod');
  my @program = ($KFOD, 'op=disableremote');

  $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@output, @program);
  trace("kfod op=disableremote rc: $rc");
  if (0 != $rc)
  {
    trace("Failed to disable remote ASM, error: @output");
    return FAILED;
  }
  else
  {
    trace("Successfully disabled remote ASM");
  }

  return SUCCESS;
}

=head2 copyASMCredentials

  Copy ASM credentials to the gpnp dir from the location specified by
  ASM_CREDENTIALS 

=head3 Parameters

  None

=head3 Returns

  SUCCESS or FAILED

=head3 Usage

=cut

sub copyASMCredentials
{
  my $srcfile = $CFG->params('ASM_CREDENTIALS');
  my $dstfile = get_asm_cred_file();
 
  my $ret = copy_file($srcfile, $dstfile,
               $CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'));

  return $ret;
}

=head2 isLegacyASM 

  Check if legacy ASM is configured

=head3 Parameters

  None

=head3 Returns

  TRUE : Legacy ASM
  FALSE: Non-legacy ASM

=head3 Usage

=cut

sub isLegacyASM
{
  my $asmConf = $CFG->params('ASM_CONFIG');
  trace("ASM_CONFIG value: $asmConf");

  if ((! $asmConf) || ((! isNearASM()) && (! isFarASM())))
  {
    trace("ASM_CONFIG is legacy");
    return TRUE;
  }

  return FALSE;
}

=head2 isFarASM

  Check if far ASM is configured 

=head3 Parameters

  None

=head3 Returns

  TRUE : Far ASM configured
  FALSE: Far ASM not configured 

=head3 Usage

=cut

sub isFarASM
{
  if ("far" eq lc($CFG->params('ASM_CONFIG')))
  {
    trace("ASM_CONFIG is far");
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}

=head2 validateLogical 

  Check if the current configuration is valid 

=head3 Parameters

  None

=head3 Returns

  Bail out if configuration is invalid

=head3 Usage

=cut

sub validateLogical
{
  # BC supports only near ASM
  if (isBigCluster() && !isNearASM())
  {
    die(dieformat(370));
  }
}

sub isNearASM
{
  # return true if ASM_CONFIG = "near"
  if ("near" eq lc($CFG->params('ASM_CONFIG')))
  {
    trace("ASM_CONFIG is near");
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}

#-------------------------------------------------------------------------------
# Function: Enables bold print if platform is not windows.
# Args    : none
# Returns : none
#-------------------------------------------------------------------------------
sub set_bold
{
  my $platform = s_get_platform_family();

  if ($platform ne 'windows' && !$CFG->AUTO) {
    print color 'bold';
  }
}

#-------------------------------------------------------------------------------
# Function: Disabled bold print if platform is not windows.
# Args    : none
# Returns : none
#-------------------------------------------------------------------------------
sub reset_bold
{
  my $platform = s_get_platform_family();

  if ($platform ne 'windows' && !$CFG->AUTO) {
    print color 'reset';
  }
}

sub rscPreChecks
#---------------------------------------------------------------------
# Function: Pre-checks for running root script 
# Args    : None
# Returns : None
#---------------------------------------------------------------------
{
  trace("Performing few checks before running scripts");

  if ($CFG->platform_family eq "unix")
  {
    my @capout = ();
    my $rc;
    my $user = $CFG->params('ORACLE_OWNER');

    trace("Attempt to get current working directory");
    $rc = run_as_user2($user, \@capout, 'pwd');
    if (0 != $rc)
    {
      trace("Failed to get current working directory: @capout");
      
      my $safeDir = $CFG->params('ORACLE_HOME');
      trace("Change working directory to safe directory $safeDir");
      chdir($safeDir) or die("Unable to change directory to $safeDir: $!\n"); 
    }
  }

  trace("Pre-checkes for running scripts passed");
}

=head2 setConfiguredCRSHome 

  Try to retrieve the configured CRS home from which the current stack
  is running, and set $CFG->configuredCRSHome

=head3 Parameters

  None

=head3 Returns

  Configured CRS home

=head3 Usage

=cut

sub setConfiguredCRSHome
{
  if (! $CFG->UPGRADE)
  {
    $CFG->configuredCRSHome($CFG->ORA_CRS_HOME);
    return $CFG->ORA_CRS_HOME;
  }

  my $home;

  trace("Get the configured CRS home from olr.loc");
  $home = s_get_olr_file("crs_home");

  if (! $home)
  {
    # pre-11.2
    trace("Get the configured CRS home from init.cssd");
    $home =  s_getOldCrsHome();
  }

  trace("Configured CRS home: ${home}");

  $CFG->configuredCRSHome($home);
  return $home;
}

=head2 getConfiguredCRSHome 

  The function for exporting the configured CRS home 

=head3 Parameters

  None

=head3 Returns

  Configured CRS home

=head3 Usage

=cut

sub getConfiguredCRSHome
{
  return $CFG->configuredCRSHome;
}

=head1 EXPORTED FUNCTIONS

=head2 startFullStack

  Start Oracle CRS stack in blocking mode. This subroutine can only work
  in post-12.1, and it should be able to work for the following two cases:
    1) The stack has been completely up
    2) The stack has been partially started and running

=head3 Parameters

  CRS home location. If undef, then current home is used. 

=head3 Returns

  SUCCESS - The stack started successfully.
  FAILED  - Failed to start the stack/stack is partially up.
  WARNING - Stack failed to start due to CRS-1665 (wrong node role).

=cut

sub startFullStack
{
  my $crshome = $_[0];
  my $role = $_[1];

  my $success = FAILED;
  my $CRSCTL;
  if (! $crshome) {
    $CRSCTL = crs_exec_path('crsctl');
  } else {
    $CRSCTL = catfile($crshome, 'bin', 'crsctl');
  }

  if ($role)
  {
    trace("Starting the stack in $role role");
  }
  my @output1 = system_cmd_capture($CRSCTL, 'start', 'crs', '-wait');
  my $rc = shift @output1;

  trace("The return value of blocking start of CRS: $rc");
  if(0 == $rc)
  {
    my @output2 = system_cmd_capture($CRSCTL, 'check', 'crs');
    $rc = shift @output2;
    if ((0 == $rc) &&
        (scalar(grep(/4537/, @output2)) > 0) &&
        (scalar(grep(/4529/, @output2)) > 0) &&
        (scalar(grep(/4533/, @output2)) > 0))
    {
      $success = SUCCESS; 
      trace("Oracle CRS stack completely started and running");
      print_lines(@output1);
    }
  }
  else
  {
    if (scalar(grep(/CRS-1665/, @output1)) > 0)
    {
      trace("Failed to start the stack in $role mode");
      $success = WARNING;
     
    }
    else
    {
      trace("Failed to start the Oracle CRS stack");
      print_lines(@output1);
    }
  }

  return $success;
}

=head1 EXPORTED FUNCTIONS

=head2 startExclCRS

  Start Oracle CRS stack in exclusive mode. This subroutine can only work
  in post-12.1.

=head3 Parameters

  None

=head3 Returns

  TRUE   - The stack is completely up.
  FALSE  - Failed to start the stack/stack is partially up.

=cut

sub startExclCRS
{
  my $is_up = FALSE;
  my $CRSCTL = crs_exec_path('crsctl');
  my $rc = system_cmd($CRSCTL, 'start', 'crs', '-excl');

  trace("The return value of blocking start of CRS: $rc");

  if(0 == $rc)
  {
    my @output = system_cmd_capture($CRSCTL, 'check', 'crs');
    $rc = shift @output;
    if ((0 == $rc) &&
        (scalar(grep(/4638/, @output)) > 0) &&
        (scalar(grep(/4692/, @output)) > 0) &&
        (scalar(grep(/4529/, @output)) > 0))
    {
      $is_up = TRUE; 
      trace("Oracle CRS stack completely started and running");
    }
  }
  else
  {
    trace("Failed to start the Oracle CRS stack in exclusive mode");
  }


  return $is_up;
}

=head1 EXPORTED FUNCTIONS

=head2 stopFullStack

  Shut down Oracle CRS stack. If the stack has been down, this func
  should return TRUE.

=head3 Parameters

  [0] "force" : Stop CRS with force option 
  [1]  CRS home location. If undef, then current home is used. 

=head3 Returns

  TURE   -  No CRS stack is running 
  FALSE  -  Failed to shut down the stack
=cut

sub stopFullStack
{
  my $force = $_[0];
  my $crshome = $_[1];

  my $is_down = FALSE;
  my $CRSCTL; 
  if (! $crshome) { 
    $CRSCTL = crs_exec_path('crsctl');
  } else {
    $CRSCTL = catfile($crshome, 'bin', 'crsctl');
  }

  my @cmd;
  if ($force eq "force")
  {
    @cmd = ($CRSCTL, 'stop', 'crs', '-f');
  }
  else
  {
    @cmd  = ($CRSCTL, 'stop', 'crs');
  }

  my @output = system_cmd_capture(@cmd);
  my $rc = shift @output;

  # If the stack has been down, the return code is non-zero.
  # We don't want the output to go to stdout because it looks
  # like a problem and probably gives rise to concerns.
  trace("The return value of stop of CRS: $rc");
  if (0 == $rc)
  {
    print_lines(@output);
  }

  @output = system_cmd_capture($CRSCTL, 'check', 'crs');
  $rc = shift @output;
  if ((0 == $rc) && (scalar(grep(/4639/, @output)) > 0))
  {
    $is_down = TRUE;
    trace("Oracle CRS stack has been shut down");
  }

  return $is_down;
}

=head1 EXPORTED FUNCTIONS

=head2 startOhasdOnly

  Start Oracle CRS stack with '-noautostart' option.

=head3 Parameters

  [0] CRS home 

=head3 Returns

  TURE or FALSE

=cut

sub startOhasdOnly
{
  my $crshome = $_[0];

  my $is_up = FALSE;
  my $CRSCTL;
  if ($crshome)
  {
    $CRSCTL = crs_exec_path('crsctl', $crshome);
  }
  else
  {
    $CRSCTL = crs_exec_path('crsctl');
  }

  my @output = system_cmd_capture($CRSCTL, 'start', 'crs', '-noautostart');
  my $rc = shift @output;

  trace("Return value of start of CRS with '-noautostart': $rc");
  if (0 == $rc)
  {
    print_lines(@output);

    @output = system_cmd_capture($CRSCTL, 'check', 'has');
    $rc = shift @output;
    if ((0 == $rc) && (scalar(grep(/4638/, @output)) > 0))
    {
      $is_up = TRUE;
      trace("Oracle High Availability Services is online");
    }

    if(!$is_up)
    {
      trace("Failed to start OHASD only");
    }
  }
  else
  {
    trace("Failed to start OHASD only");
  }

  return $is_up;
}

=head1 EXPORTED FUNCTIONS

=head2 startExclNoCRS

  Start Oracle Clusterware in exclusive mode without starting CRS

=head3 Parameters

  [0] CRS home

=head3 Returns

  TURE or FALSE

=cut

sub startExclNoCRS
{
  my $crshome = shift;
  my $is_up = FALSE;
  my $CRSCTL = catfile($crshome, 'bin', 'crsctl');

  my $rc = system_cmd($CRSCTL, 'start', 'crs', '-excl', '-nocrs');
  trace("Return value of starting CRS with '-excl' and '-nocrs' options: $rc");

  if(0 == $rc)
  {
    my @output = system_cmd_capture($CRSCTL, 'check', 'crs');
    $rc = shift @output;
    if ((0 == $rc) &&
        (scalar(grep(/4638/, @output)) > 0) &&
        (scalar(grep(/4529/, @output)) > 0))
    {
      $is_up = TRUE;
      trace("Successfully start Oracle Clusterware in exclusive mode ".
            "without CRSD running");
    }
  }
  else
  {
    trace("Failed to start Oracle Clusterware in exclusive mode with -nocrs");
  }


  return $is_up;
}

=head1 EXPORTED FUNCTIONS

=head2 createMgmtdbDir

  Create management DB directory for DBCA

=head3 Parameters

  None

=head3 Returns

  None  

=cut

sub createMgmtdbDir
{
  if (($CFG->defined_param('MGMT_DB')) &&
      ("true" eq lc($CFG->params('MGMT_DB'))) &&
      (! isOCRonASM())) 
  {
    trace("Creating management DB directory");

    my $ocr_loc = s_get_config_key("ocr", "ocrconfig_loc");
    # check if it's a symbolic link
    if (-l $ocr_loc)
    {
      $ocr_loc = s_getAbsLink($ocr_loc);
    }

    my $pdir = dirname($ocr_loc);
    if (! -e $pdir) { die(dieformat(22, $pdir)); }

    my $mgmtdb_dir = catdir($pdir, 'mgmtdb');
    create_dir($mgmtdb_dir);
    if (! -e $mgmtdb_dir)
    {
      trace("Failed to create management DB directory '$mgmtdb_dir'");
      die(dieformat(22, $mgmtdb_dir));
    } 

    s_set_ownergroup($CFG->params('ORACLE_OWNER'),
                       $CFG->params('ORA_DBA_GROUP'), 
                       $mgmtdb_dir) or die(dieformat(152, $mgmtdb_dir));

    s_set_perms("0750", $mgmtdb_dir) or die(dieformat(153, $mgmtdb_dir));

    trace("Created management DB directory successfully");
  }
  else
  {
    trace("No need to create management DB directory");
  }
}

# For checking version less than 11202

sub isVersionLT11202
{

  my $is112ver;

  my @oldcrs_ver = @{$CFG->oldconfig('ORA_CRS_VERSION')};
  my $verinfo = join('.', @oldcrs_ver);

  if (int($oldcrs_ver[0]) <= int('10')) {
      $is112ver = TRUE;
  }elsif (-1 == versionComparison($verinfo, "11.2.0.2.0"))
  {
      $is112ver = TRUE;
  }else {
      $is112ver = FALSE;
      if ($CFG->platform_family eq "windows" || 
          $^O =~ /aix/i) {
          if (-1 == versionComparison($verinfo, "11.2.0.3.0")) {
              $is112ver = TRUE;
          }
      }
  }

  return $is112ver;

}

# For checking version greater than or equal to 1201
sub isVersionGTE1201
{

  my $is121ver;

  my @crs_ver = split(/\./, getcrsrelver());

  if (int($crs_ver[0]) >= int('12')) {
      $is121ver = TRUE;
  }else {
      $is121ver = FALSE;
  }

  return $is121ver;

}

# Wrapper for checking the version in case if we want to change it back to 
# bdb we need to change wrapper.
sub isReposBDB
{
  return isVersionGTE1201();
}

=head2 get_olsnodes_info

  Gets olsnodes output for given command line params. 

=head3 Parameters

  string with olsnodes home location. If undef, then current home is used.

=head3 Returns

  =head4 returns a list of strings with node names. Warning messages, if any,
         are filtered out.
  =head4 result code (0 for success) as a first member of array.

=cut

sub get_olsnodes_info
{
   my @nodes = ();

   my ($home, @args) = @_;
   my $cmd;
   if (! defined $home) {
     $cmd = crs_exec_path('olsnodes');
   } else {
     $cmd = catfile( $home, 'bin', 'olsnodes' );
   }

   # run olsnodes w/given pars
   my @out = system_cmd_capture(($cmd, @args));
   my $rc  = shift @out;

   # read-in interface list 
   if (0 == $rc) {
      trace "---Got olsnodes out ($cmd ".join(' ',@args)."):";
      foreach (0..$#out) {
         my $node = $out[$_];
         trace $node ;
         # total failure should return rc, else filter out non-fatal
         # error messages, if any, e.g. "PRCO-nn: error....."
         if ($node !~ /^PR[A-Z]+-[0-9]+: /) {
           push @nodes, $node ;
         } else {
           trace( $node );
           print_error(174);
         }
      }
   } else {
      push @nodes, "$cmd ".join(' ',@args)." failed.";
   }
   return ($rc, @nodes);
}

=head1 EXPORTED FUNCTIONS

=head2 getNodeNumList 

  Gets a list of node numbers

=head3 Parameters

  [0] CRS home (input)

=head3 Returns

  A string that looks like:
  <node1>:<number1>,<node2>:<number2>...

  otherwise, returns undef string

=cut

sub getNodeNumList
{
  my $crshome = $_[0];
  my $rc = 0;
  my @olsnodes_out;
  my $nodeNumList;

  ($rc, @olsnodes_out) = get_olsnodes_info($crshome, '-n');
  if (0 != $rc)
  {
    trace("olsnodes -n (". join(' ' , @olsnodes_out). ")");
    return $nodeNumList;
  }

  foreach my $elem (@olsnodes_out)
  {
    chomp($elem);
    $elem =~ s/\s+/:/; 

    if ($nodeNumList)
    {    
      $nodeNumList = "$nodeNumList," . "$elem";
    }
    else
    {
      $nodeNumList = $elem;
    }
  }

  trace("Node number list: {$nodeNumList}");
  return $nodeNumList;
}

=head1 EXPORTED FUNCTIONS

=head2 getCurrentNodenameList

  Gets the list of current nodes.

=head3 Parameters

  None

=head3 Returns

  @node_list - list of nodes

=cut
       
sub getCurrentNodenameList
{
   my @node_list = split (/,/, $CFG->params('NODE_NAME_LIST'));

   return @node_list;
}

=head1 EXPORTED FUNCTIONS

=head2 getLocalNodeName 

  Gets the name of local node 

=head3 Parameters

  [0] CRS home (input)

=head3 Returns

  A string that represents the name of local node

  otherwise, returns undef string

=cut

sub getLocalNodeName
{
  my $crshome = $_[0];
  my $rc = 0;
  my @olsnodes_out;
  my $locNodeNm;

  ($rc, @olsnodes_out) = get_olsnodes_info($crshome, '-l');
  if ((0 != $rc) || (0 == scalar(@olsnodes_out)))
  {
    trace("olsnodes -l (". join(' ' , @olsnodes_out). ")");
    return $locNodeNm;
  }

  $locNodeNm = $olsnodes_out[0];
  chomp($locNodeNm);

  trace("Local node name: {$locNodeNm}");
  return $locNodeNm;
}

=head1 EXPORTED FUNCTIONS

=head2 isASMConfigured

  Check if ASM configured

=head3 Parameters

  [0] CRS home (input)

=head3 Returns

  TRUE  if     configured
  FALSE if not configured

=cut

sub isASMConfigured
{
  my $crshome = $_[0];
  my $crsctl = catfile($crshome, 'bin', 'crsctl');
  my @output = system_cmd_capture($crsctl, "stat", "res", "-w",
                 "\"TYPE = ora.diskgroup.type\"");
  my $rc = shift @output;

  if (0 != $rc)
  {
    die(dieformat(180, "crsctl stat res", $rc));
  }
  
  if (0 == scalar(@output))
  {
    trace("ASM is not configured");
    return FALSE;
  }

  trace("ASM is configured");
  return TRUE;
}

sub add_localOlr_OlrConfig_OcrConfig
#-------------------------------------------------------------------------------
# Function: add local olr to crsconfig_fileperms file
# Args    : none
#-------------------------------------------------------------------------------
{
   my $SUPERUSER     = $CFG->SUPERUSER;
   my $ORACLE_OWNER  = $CFG->params('ORACLE_OWNER');
   my $ORA_DBA_GROUP = $CFG->params('ORA_DBA_GROUP');
   my $platform      = $CFG->platform_family;
   my $OLR_LOCATION  = $CFG->OLR_LOCATION;
   my ($OCRCONFIG, $OLRCONFIG);

   # open crsconfig_fileperms
   my $permsfile = catfile($CFG->ORA_CRS_HOME, 'crs', 'utl', 'crsconfig_fileperms');
   open (FPFILE, ">>$permsfile") || die(dieformat(208, $permsfile, $!));

   # add OLR_LOCATION
   if ($CFG->SIHA) {
      print FPFILE
            "$platform $OLR_LOCATION $ORACLE_OWNER $ORA_DBA_GROUP 0600\n";
   }
   else {
      print FPFILE
            "$platform $OLR_LOCATION $SUPERUSER $ORA_DBA_GROUP 0600\n";
   }

   if ($platform eq "unix") {
      $OCRCONFIG = $CFG->params('OCRCONFIG');
      $OLRCONFIG = $CFG->params('OLRCONFIG');

      # add OLRCONFIG
      if ($OLRCONFIG) {
         if (is_dev_env()) {
            print FPFILE "$platform $OLRCONFIG " .
                         "$ORACLE_OWNER $ORA_DBA_GROUP 0644\n";
         } else { 
            print FPFILE "$platform $OLRCONFIG " .
                         "$SUPERUSER $ORA_DBA_GROUP 0644\n";
         }
      }

      # add OCRCONFIG
      if (! $CFG->SIHA) {
         if ($OCRCONFIG) {
            print FPFILE
                  "$platform $OCRCONFIG $SUPERUSER " .
                   "$ORA_DBA_GROUP 0644\n";
         }
      }
   }

   close (FPFILE);
}

=head1 EXPORTED FUNCTIONS

=head2 extractDiskgroups

  Querying CSS vote disks for getting DGs  

=head3 Parameters

  None

=head3 Returns

  An array containing DGs

=cut
sub extractDiskgroups
{
  my @dglist;

  my @css_votedisk = queryVoteDisks();

  foreach my $votedisk (@css_votedisk)
  {
    trace("Voting disk is : $votedisk");
    if ($votedisk =~ / \[(.+)\]/)
    {
      # E.g., $votedisk is:
      #   '1.  2 282bf2a833f54f02bf4befd002fa90d6(/dev/raw/raw1) [OCRDG]'
      # parse $votedisk to get 'OCRDG'
      push (@dglist, $1);
    }
  }

  trace("Diskgroups found: @dglist");
  return @dglist;
}

=head1 EXPORTED FUNCTIONS

=head2 removeVotingDiks
  
  Delete voting file(s) from the CSSD configuration when OCR is on ASM
    
=head3 Parameters

  [0] CRS home
  [1] ASM diskgroup to store voting file(s)
  
=head3 Returns
  
  SUCCESS or FAILED 
  
=cut

sub removeVotingDiks
{
  my $crshome        = $_[0];
  my $ASM_DISK_GROUP = $_[1];
  my $crsctl;
  my $cmd;
  my @output;
  my $rc;
  my $dgname;

  if (! $crshome)
  {
    trace("Invalid CRS home");
    return FAILED;
  }

  $crsctl = catfile($crshome, "bin", "crsctl");
  $dgname = $ASM_DISK_GROUP;
  if ($CFG->platform_family ne "windows")
  {
    $ASM_DISK_GROUP = "'+$ASM_DISK_GROUP'";
  }
  else
  {
    $ASM_DISK_GROUP = "+$ASM_DISK_GROUP";
  }

  $cmd = "$crsctl delete css votedisk $ASM_DISK_GROUP";
  @output = system_cmd_capture($cmd);
  $rc = shift @output;

  if (0 != $rc)
  {
    trace("crsctl delete for vds in $dgname ... failed");
    return FAILED;
  }

  return SUCCESS;
}

=head1 EXPORTED FUNCTIONS

=head2 addVotingDisks
  
  Add voting file(s) to the CSSD configuration when OCR is on ASM
    
=head3 Parameters

  [0] CRS home
  [1] ASM diskgroup to store voting file(s)
  
=head3 Returns
  
  SUCCESS or FAILED 
  
=cut

sub addVotingDisks
{
  my $crshome        = $_[0];
  my $ASM_DISK_GROUP = $_[1];
  my $crsctl;
  my $addvfcmd;
  my $dgname;

  if (! $crshome)
  {
    trace("Invalid CRS home");
    return FAILED;
  }

  $crsctl = catfile($crshome, 'bin', 'crsctl');
  $dgname = $ASM_DISK_GROUP;
  if ($CFG->platform_family ne "windows")
  {
    $ASM_DISK_GROUP = "'+$ASM_DISK_GROUP'";
  }
  else
  {
    $ASM_DISK_GROUP = "+$ASM_DISK_GROUP";
  }

  trace("Creating voting files on ASM diskgroup $dgname");
  $addvfcmd = "$crsctl replace votedisk $ASM_DISK_GROUP";
  my @output = system_cmd_capture($addvfcmd);
  my $rc = shift @output;

  if (0 != $rc)
  {
    trace("Voting disks add failed");
    return FAILED;
  }

  return SUCCESS;
}

=head1 EXPORTED FUNCTIONS

=head2 removeVotingfiles
  
  Delete voting file(s) from the CSSD configuration when OCR is on filesystem
    
=head3 Parameters

  [0] CRS home
  [1] Voting files(s) to be deleted
  
=head3 Returns
  
  SUCCESS or FAILED 
  
=cut

sub removeVotingfiles
{
  my $crshome      = $_[0];
  my $VOTING_FILES = $_[1];
  my $crsctl;
  my $delvfcmd;

  if (! $crshome)
  {
    trace ("Invalid CRS home");
    return FAILED;
  }

  $crsctl = catfile($crshome, 'bin', 'crsctl');
  $delvfcmd = "$crsctl delete css votedisk @$VOTING_FILES";

  my @output = system_cmd_capture($delvfcmd);
  my $rc = shift @output;
  if (0 != $rc)
  {
    error("Voting files deletion failed");
    return FAILED;
  }

  trace("Voting files deletion succeeded");
  return SUCCESS;
}

=head1 EXPORTED FUNCTIONS

=head2 addVotingFiles
  
  Add voting file(s) to the CSSD configuration when OCR is on filesystem
    
=head3 Parameters

  [0] CRS home
  [1] Voting files(s) to be added 
  
=head3 Returns
  
  SUCCESS or FAILED 
  
=cut

sub addVotingFiles
{
  my $crshome      = $_[0];
  my $VOTING_FILES = $_[1];
  my $crsctl;
  my $addvfcmd;

  if (! $crshome)
  {
    trace("Invalid CRS home");
    return FAILED;
  }

  $crsctl = catfile($crshome, 'bin', 'crsctl');
  $addvfcmd = "$crsctl add css votedisk @$VOTING_FILES";

  my @output = system_cmd_capture($addvfcmd);
  my $rc = shift @output;
  if ($rc != 0)
  {
    trace("Voting files add failed");
    return FAILED;
  }
 
  trace("Voting files add succeeded");
  return SUCCESS;
}

=head1 EXPORTED FUNCTIONS

=head2 start_crsd_and_check
  
  Start resource ora.crsd and check if it is online
    
=head3 Parameters

  [0] CRS home
  
=head3 Returns
  
  SUCCESS or FAILED 
  
=cut

sub start_crsd_and_check
{
  my $crshome = shift;
  
  my $cmd;
  my @chk;
  my $rc = -1;
  my $retries = 120;
  my $crsctl = catfile($crshome, 'bin', 'crsctl');
  
  $cmd = "$crsctl start resource ora.crsd -init";
  trace("Executing \'$cmd\'");
  system_cmd("$cmd");

  $cmd = "$crsctl check crs";
  while ($retries--)
  {
    @chk = system_cmd_capture($cmd);
    $rc = shift @chk;
    
    if ((0 == $rc) &&
         ((scalar(grep(/4537/, @chk)) > 0) ||
          (scalar(grep(/4692/, @chk)) > 0)))
    {
      trace("The resource ora.crsd is online");
      last;
    }

    sleep(5);
  } 

  if ($retries > 0)
  { 
    return SUCCESS;
  } 
  else
  {
    return FAILED;
  }
}

=head1 EXPORTED FUNCTIONS

=head2 start_excl_crs

  Start Oracle CRS stack in exclusive mode

=head3 Parameters

  [0] CRS home 

=head3 Returns

  TRUE or FALSE

=cut

sub start_excl_crs
{
  my $crshome = shift;

  my $cmd;
  my @output;
  my $rc = -1;
  my $is_up = FALSE;
  my $crsctl = catfile($crshome, 'bin', 'crsctl');

  $cmd = "$crsctl start crs -excl";
  trace("Executing \'$cmd\'");
  $rc = system_cmd("$cmd");

  trace("The return value of start of CRS exclusive mode: $rc");

  $cmd = "$crsctl check crs";
  @output = system_cmd_capture($cmd);
  $rc = shift @output; 
  if ((0 == $rc) &&
      (scalar(grep(/4638/, @output)) > 0) &&
      (scalar(grep(/4692/, @output)) > 0) &&
      (scalar(grep(/4529/, @output)) > 0))
  {
    $is_up = TRUE;
    trace("Cluster Ready Services is online in exclusive mode");
  }

  return $is_up;
}

=head1 EXPORTED FUNCTIONS
  
=head2 stop_crsd_and_check
  
  Stop resource ora.crsd and check if it is offline 
    
=head3 Parameters

  [0] CRS home
  
=head3 Returns
  
  SUCCESS or FAILED 
  
=cut 

sub stop_crsd_and_check
{
  my $crshome = shift;

  my $cmd;
  my @chk;
  my $stop_rc = -1;
  my $rc = -1;
  my $crsctl = catfile($crshome, 'bin', 'crsctl');

  $cmd = "$crsctl stop resource ora.crsd -init -f";
  trace("Executing \'$cmd\'");
  $stop_rc = system_cmd("$cmd");

  trace("The return value of stop of ora.crsd: $stop_rc");

  $cmd = "$crsctl check crs";
  @chk = system_cmd_capture($cmd);
  $rc = shift @chk;
  if ((0 == $rc) && (scalar(grep(/4535/, @chk)) > 0))
  {
    return SUCCESS;
  }
  else
  {
    return FAILED;
  }
}

sub upgradeModelResType
#-------------------------------------------------------------------------------
# Function:  Creates the resource types for all nodeapps and the crsd managed
#            resources. 
# Args    :  none
# Returns :  TRUE  if 'srvctl upgrade model -restype' succeeds
#            FALSE if 'srvctl upgrade model -restype' fails 
#-------------------------------------------------------------------------------
{
  my $run_as_owner = FALSE;
  my $success = TRUE;
  my $status = srvctl($run_as_owner,
                          "upgrade model -restype");
   if (${status}) {
      trace ("srvctl upgrade model -restype  ... passed");
   }
   else {
      print_error(180, "srvctl upgrade model -restype", ${status});
      $success = FALSE;
   }

   return $success;
}

sub isVersionMatch
{
   my $ver1 = $_[0];
   my $ver2 = $_[1];
   my $status;
   chomp $ver1;
   chomp $ver2;
   trace("version1 is $ver1");
   trace("version2 is $ver2");

   if ($ver1 =~ $ver2) {
      $status = TRUE;
   }
   else {
      $status = FALSE;
   }

   trace("Version match status is $status");
   return $status;
}

sub check_NewCrsStack
#------------------------------------------------------------------------------- 
# Function: Check if the stack is up from current CrsHome
# Args    : none                                                    
#------------------------------------------------------------------------------- 
{                  
   # Forward merge fix for bug 12640884 
   # - LNX64-11203-UD: CRSCTL CHECK CSSD HANG, 10.2.0.5 -> 11.2.0.3 
   # Modify the check_NewcrsStack routine to not use the crsctl check command    
   # for cssd, evmd and crsd and only use the check_service routine to get
   # the status
                                                                   
   trace("check new crs stack");
   my $status = TRUE;
   my $new_crshome = $CFG->ORA_CRS_HOME;
   
   if ((!check_service("ora.cssd", 3)) &&             
        (!check_service("ora.crsd", 3)) &&
        (!check_service("ora.evmd", 3)))  
   {                            
      $status = FALSE;
   }                                 
     
   if ($status)
   { 
      trace ("Oracle Grid Infrastructure is running from $new_crshome");
   } 
   else {
      trace ("Oracle Grid Infrastructure is not running from $new_crshome");     
   }                 
 
   return $status;
}

=head1 EXPORTED FUNCTIONS

=head2 getNodeStatus
  
  Gets the status of each node in the cluster
 
=head3 Parameters
  
  [0] CRS home  
  
=head3 Returns
    
  A hash array containing active/inactive nodes.
  The active nodes are marked with '1'. 
    
=cut

sub getNodeStatus
{
  my $crshome = $_[0];
  my $rc = 0;
  my @olsnodes_out;
  my %nodeStatus;

  ($rc, @olsnodes_out) = get_olsnodes_info($crshome, '-s');
  if (0 != $rc)
  {
    die "Cannot get active nodes (" . join(' ',@olsnodes_out). ")";
  }

  foreach my $nodeinfo (@olsnodes_out)
  {
    if ($CFG->DEBUG) { trace("nodeinfo: $nodeinfo"); }

    my @nodeinf = split(/\s+/, $nodeinfo);
    if ($nodeinf[1] =~ /Active/)
    {
      $nodeStatus{${nodeinf[0]}} = 1;
    }
    else
    {
      $nodeStatus{${nodeinf[0]}} = 0;
    }
  }

  return %nodeStatus;
}

=head1 EXPORTED FUNCTIONS

=head2 getActiveNodeRoles
  
  Gets active node roles of the nodes in the cluster
 
=head3 Parameters
  
  [0] CRS home  
  
=head3 Returns
    
  A hash array containing active node roles.
    
=cut

sub getActiveNodeRoles
{
  my $crshome = $_[0];
  my $rc = 0;
  my @olsnodes_out;
  my %nodeRoles;

  ($rc, @olsnodes_out) = get_olsnodes_info($crshome, '-a');
  if (0 != $rc)
  {
    trace("Cannot get active node roles(" . join(' ',@olsnodes_out). ")"); 
    return %nodeRoles;
  }

  foreach my $nodeinfo (@olsnodes_out)
  {
    trace("nodeinfo: $nodeinfo");
    my @nodeinf = split(/\s+/, $nodeinfo);
    $nodeRoles{${nodeinf[0]}} = lc($nodeinf[1]);
  }

  return %nodeRoles;
}

sub add_rim_listener
#-------------------------------------------------------------------------------
# Function:  Adds the rim listener for a Big Cluster.
# Args    :  none
# Returns :  TRUE if 'srvctl add listener -leaflistener' succeeds
#            FALSE if 'srvctl add listener -leaflistener' fails
#-------------------------------------------------------------------------------
{
   my $success = TRUE;

   if(isBigCluster()) {
      my $run_as_owner = TRUE;
      my $status = srvctl($run_as_owner, "add listener -leaflistener");

      if (${status}) {
         trace ("srvctl add listener -leaflistener  ... passed");
      }
      else {
         print_error(180, "srvctl add listener -leaflistener", ${status});
         $success = FALSE;
      }
   }

   return $success;
}

=head1 EXPORTED FUNCTIONS

=head2 collect_cvu_baseline

  Collect CVU baseline

=head3 Parameters

  [0] baseline report name

=head3 Returns

  None

=cut

sub collect_cvu_baseline
{
  my $report_file = $_[0];
  my $node_list = $CFG->params('NODE_NAME_LIST');

  if (! isCkptexist("ROOTCRS_CVU_BASELINE_COLLECT"))
  {
    trace("Write a checkpoint for CVU baseline collection");
    writeCkpt("ROOTCRS_CVU_BASELINE_COLLECT", CKPTSTART);
  }

  if (! isCkptSuccess("ROOTCRS_CVU_BASELINE_COLLECT"))
  {
    $CFG->wipCkptName("ROOTCRS_CVU_BASELINE_COLLECT");
    if (SUCCESS == exec_cvu_baseline_collect($report_file, $node_list))
    {
      writeCkpt("ROOTCRS_CVU_BASELINE_COLLECT", CKPTSUC);
    }
    else
    {
      writeCkpt("ROOTCRS_CVU_BASELINE_COLLECT", CKPTFAIL);
    }

    $CFG->wipCkptName("ROOTCRS_STACK");
  }
}

=head2 exec_cvu_baseline_collect

  The subroutine for executing CVU baseline collect command

=head3 Parameters

  [0] baseline report name
  [1] the comma-separated list of node names

=head3 Returns

  SUCCESS or FAILED

=cut

sub exec_cvu_baseline_collect
{
  my $report_file = $_[0];
  my $node_list = $_[1];

  trace("Collecting CVU baseline ...");

  my $cluvfy = crs_exec_path('cluvfy');
  my $cmd = "$cluvfy comp baseline -collect cluster -n $node_list -rootscript -saveToAllNodes -reportname $report_file";
  my $rc = run_as_user($CFG->params('ORACLE_OWNER'), $cmd);

  if (0 != $rc)
  {
    trace("Failed to collect CVU baseline"); 
    return FAILED;
  }

  return SUCCESS;
}

=head1 EXPORTED FUNCTIONS

=head2 checkGIStack

  Check CRS stack health

=head3 Parameters

  CRS home location. If undef, then current home is used. 

=head3 Returns
 
  GI_STACK_DOWN   : CRS stack is completely down
  GI_STACK_UP     : CRS stack is fully up
  GI_STACK_PARTIAL: CRS stack is partially running

=cut

sub checkGIStack
{
  my $crshome = $_[0];

  my $stat = GI_STACK_DOWN;
  my $CRSCTL;
  if (! $crshome) {
    $CRSCTL = crs_exec_path('crsctl');
  } else {
    $CRSCTL = catfile($crshome, 'bin', 'crsctl');
  }

  my @output = system_cmd_capture($CRSCTL, 'check', 'crs');
  my $rc = shift @output;
  if ((0 == $rc) &&
      (scalar(grep(/4537/, @output)) > 0) &&
      (scalar(grep(/4529/, @output)) > 0) &&
      (scalar(grep(/4533/, @output)) > 0))
  {
    $stat = GI_STACK_UP;
    trace("Oracle CRS stack is completely up");
  }

  return $stat;
}

=head1 EXPORTED FUNCTIONS

=head2 genTimeStamp
  
  Generates a time stamp in the format YYYY-MM-DD_hh-mm-ss(AM|PM)
 
=head3 Parameters
  
  None
  
=head3 Returns
    
  A string containing the time stamp.
    
=cut

sub genTimeStamp
{
   my ($sec, $min, $hour, $day, $month, $year) = (localtime) [0, 1, 2, 3, 4, 5];
   my $timeOfDay = "AM";

   $year = $year + 1900;
   $month = $month + 1;
   if ($hour > 12) {
      $hour -= 12;
      $timeOfDay = "PM";
   }

  return sprintf("%04d-%02d-%02d_%02d-%02d-%02d%s", $year, $month, $day, $hour, $min, $sec, $timeOfDay);
}

sub stop_ohasd_resources
{
  my $rc = FAILED;
  trace("Attempt to stop CRS stack and then start OHASD only");

  if (stopFullStack("force"))
  {
    $rc = SUCCESS;
    trace("Succeeded in shutting down Oracle CRS stack");
  }

  if ((SUCCESS == $rc) && startOhasdOnly())
  {
    $rc = SUCCESS;
    trace("Oracle High Availability Services is online");
  }

  return $rc;
}

sub isReusedg
{
  my $isreusedg = FALSE;
  if ($CFG->params('REUSEDG') =~ m/true/i) {
      $isreusedg = TRUE;
   }
  trace("Reuse Disk Group is set to $isreusedg");
  return $isreusedg;
}

=head1 EXPORTED FUNCTIONS

=head2 isFirstNodeToConfig

  Check if the local node is the first node to be configured:
  If RC, the first node should be the first one in NODE_NAME_LIST.
  If BC, the first node should be the first one in HUB_NODE_LIST or 
  AUTO_NODE_LIST.

=head3 Parameters

  [0] The name of local node [IN]
  [1] The node name of the first node to be configured [OUT]

=head3 Returns
 
  TRUE or FALSE
  The OUT paramter [1] to return the right node name if FALSE

=cut

sub isFirstNodeToConfig
{
  my $localNode = $_[0];

  my @nodeList;
  my $ret;

  if (isAddNode($localNode, $CFG->params('NODE_NAME_LIST')))
  {
    trace("An add node scenario");
    return FALSE; 
  }

  if (isBigCluster())
  {
    if (! $CFG->params('HUB_NODE_LIST'))
    {
      @nodeList = split(/,/, $CFG->params('AUTO_NODE_LIST'));
    }
    else
    {
      @nodeList = split(/,/, $CFG->params('HUB_NODE_LIST'));
    }
  }  
  else
  {
    @nodeList = split(/,/, $CFG->params('NODE_NAME_LIST'));
  }

  $ret = ($nodeList[0] =~ /\b$localNode\b/i) ? TRUE : FALSE;
  $_[1] = ($ret == TRUE) ? "" : $nodeList[0];
  return $ret;
}

#-------------------------------------------------------------------------------
# Function:  Prints the lines received to the console
# Args    :  Lines to print
# Returns :  none
#-------------------------------------------------------------------------------
sub print_lines
{
  output_lines(TRUE, FALSE, @_);
}

#-------------------------------------------------------------------------------
# Function:  Prints the lines received to the trace file
# Args    :  Lines to print
# Returns :  none
#-------------------------------------------------------------------------------
sub trace_lines
{
  output_lines(FALSE, TRUE, @_);
}

#-------------------------------------------------------------------------------
# Function:  Prints the lines received to the console and to the trace file
# Args    :  Lines to print
# Returns :  none
#-------------------------------------------------------------------------------
sub print_trace_lines
{
  output_lines(TRUE, TRUE, @_);
}

#-------------------------------------------------------------------------------
# Function:  Prints the lines received to the console and/or to the trace file
# Args    :  [0] Boolean stating if the lines should be printed to the console
#            [1] Boolean stating if the lines should be sent to the trace file
#            [2..n] Lines to print
# Returns :  none
#-------------------------------------------------------------------------------
sub output_lines
{
  my $print = shift @_;
  my $trace = shift @_;

  foreach my $line (@_)
  {
    if ($print)
    {
      print "$line\n";
    }
    if ($trace)
    {
      trace($line);
    }
  }
}

##  The same log level setting support is used in the tests.
 
# To set logging levels during startup, you can define an env variable
# that specifies both deamons and components.  The environment variable
# is HAS_DAEMON_LOGGING and the format of the variable is:
#   <daemon>#<comp-str>[~<daemon>#<comp-str>[~...]]
# For example:
#   ctss#CSSCLNT=5,CRSCCL=3
# will generate the following command:
#   crsctl set log ctss "CSSCLNT=5,CRSCCL=3"
# and:
#  ctss#CSSCLNT=5,CRSCCL=3~css#CSSD=5
# will generate the following commands:
#   crsctl set log ctss "CSSCLNT=5,CRSCCL=3"
#   crsctl set log css  "CSSD=5"
sub set_logging {
  my $crsctl = catfile($CFG->ORA_CRS_HOME, 'bin', 'crsctl');
  my @comps  = split('~', $ENV{'HAS_DAEMON_LOGGING'});
  for my $compspec (@comps) {
    my ($compname, $logset) = split('#', $compspec);

    trace("Setting logging levels of $compname to: $logset");
    system_cmd_capture($crsctl, 'set', 'log', $compname, "\"$logset\"");
  }
}

# Bounce the ohasd so that the ohasd resources/types take effect
# This also makes sure the ora.drivers.acfs or ora.asm is not online,
# otherwise, 'acfsroot install' would fail to unload the driver that
# is still in use in case of rerun.
#
# This must be between create_ohasd_resources and perform_installUSMDriver
sub bounce_ohasd
{
  trace("Attempt to bounce ohasd"); 
  stopFullStack("force") || die(dieformat(349));
  startOhasdOnly() || die(dieformat(117));  
}

sub getcrsrelver
{
   my $home = $_[0];
   my ($crsctl);
   my $verstring;                                                          

   if (! defined $home) {                                                  
      $crsctl = crs_exec_path('crsctl');
   } else {                                                                
      $crsctl = catfile($home, 'bin', 'crsctl' );
   }

   my @cmd = ($crsctl, 'query', 'crs', 'releaseversion');                  
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;                                                   

   # if succeeded, parse to ver numbers, output must be a single line,     
   # version is 5 numbers, major to minor (see above)
   if ($rc == 0) {                                                         
      $verstring = getVerInfo($out[0]);
      trace( "Got CRS release version: ".join('.', $verstring) );          
   }
   else {                                                                  
      
      error ("@cmd ... failed rc=$rc with message:\n @out \n");            
   }
   return $verstring;                                                      

}

#-------------------------------------------------------------------------------
# Function:  Returns the path to the qosctl script
# Args    :  none
# Returns :  Path to the qosctl script
#-------------------------------------------------------------------------------
sub get_qosctl_path {
  my $pgm = $CFG->s_get_qosctl_path_p;
  return &$pgm();
}

=head1 EXPORTED FUNCTIONS

=head2 isLastNodeToConfig
  
  Check if the current node is the last node to be configured by root.sh 
 
=head3 Parameters
  
  None
  
=head3 Returns
    
  TRUE or FALSE
    
=cut

sub isLastNodeToConfig
{
  my @cmd;
  my @out;
  my $rc;
  my $olsnodes = catfile($CFG->ORA_CRS_HOME, 'bin', 'olsnodes');
  my @clunodes = split(',', $CFG->params('NODE_NAME_LIST'));

  if (isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST')))
  {
    trace("Node being added is a last node");
    return TRUE;
  }

  @cmd = ($olsnodes);
  @out = system_cmd_capture(@cmd);
  $rc = shift(@out);

  if ((0 == $rc) && (scalar(@out) == scalar(@clunodes)))
  {
    trace("Last node to configure");
    return TRUE;
  }

  return FALSE;
}

=head1 EXPORTED FUNCTIONS

=head2 getLsnrUsername 
  
  Retrieve the listener username
 
=head3 Parameters
  
  None
  
=head3 Returns
    
  Listener username   
    
=cut

sub getLsnrUsername 
{
  my $lsnrUname;

  if (($CFG->defined_param('LISTENER_USERNAME')) &&
       $CFG->params('LISTENER_USERNAME') &&
       (($CFG->params('LISTENER_USERNAME')) !~ /^%/))
  {
    $lsnrUname = $CFG->params('LISTENER_USERNAME');
  }
  else
  {
    $lsnrUname = $CFG->params('ORACLE_OWNER');
  }

  return $lsnrUname;
}

=head1 EXPORTED FUNCTIONS

=head2 createNetLsnrWithUsername 
  
  Create a network listener with given user name
 
=head3 Parameters
  
  [0] The owner of listener
  [1] Listener name. When not specified, it defaults to "LISTENER" 
  
=head3 Returns
    
  None 
    
=cut

sub createNetLsnrWithUsername
{
  my $lsnr_username = $_[0];
  my $lsnr_name = $_[1];

  $lsnr_name = ($lsnr_name) ? ($lsnr_name) : "LISTENER";

  if (isFirstNodeToStart())
  {
    my $run_as_owner;
    my $status;

    if ($lsnr_username ne ($CFG->params('ORACLE_OWNER')))
    {
      trace("Create Net Listener '$lsnr_name' with the username '$lsnr_username'");
      $run_as_owner = FALSE; # run as root
      $status = srvctl($run_as_owner,
                          "add listener -l ${lsnr_name} -user ${lsnr_username}");
    }
    else
    {
      trace("Create Net Listener '$lsnr_name' as GI user");
      $run_as_owner = TRUE; # run as Oracle owner
      $status = srvctl($run_as_owner,
                          "add listener -l ${lsnr_name}");
    }

    my $traceuser = ($lsnr_username ne ($CFG->params('ORACLE_OWNER'))) ? "-u $lsnr_username" : "";

    if (TRUE == $status)
    {
      trace("add listener -l $lsnr_name " . "$traceuser" . " ... passed");
    }
    else
    {
      die(dieformat(180,
            "srvctl add listener -l $lsnr_name" . " $traceuser", $status));
    }
  }
}

=head1 EXPORTED FUNCTIONS

=head2 stopNetLsnr 
  
  Stop the given node listener
 
=head3 Parameters
  
  [0] GI home
  [1] Listener name. When not specified, it defaults to "LISTENER" 
  
=head3 Returns
    
  None 
    
=cut

sub stopNetLsnr
{
  my $crs_home  = $_[0];
  my $lsnrName = $_[1];

  $lsnrName = ($lsnrName) ? ($lsnrName) : "LISTENER";
  trace("Stopping node listener '$lsnrName' ...");

  my $lsnr_name;
  my @clunodes = split(',', $CFG->params('NODE_NAME_LIST'));
  foreach my $node (@clunodes)
  {
    if (isOldVersionLT112())
    {
      $lsnr_name = $lsnrName."_".uc($node);
      trace("Created pre-11.2 style listener name: $lsnr_name");

      my $srvctl = catfile($crs_home, 'bin', 'srvctl');
      my $env = $ENV{'SRVM_TRACE'};
      undef $SRVM_TRACE;
      my @output = system_cmd_capture($srvctl, 'stop', 'listener', '-n', $node, '-l', $lsnr_name);
      $ENV{'SRVM_TRACE'} = $env;
      my $rc = shift(@output);

      if ((0 == $rc) || (2 == $rc))
      {
        trace("Successfully stopped listener '$lsnr_name' on node '$node'");
      }
      else
      {
        die(dieformat(180,
             "srvctl stop listener -n $node -l $lsnr_name", "null"));
      }
    }
    else
    {
      $lsnr_name = $lsnrName;
      my $cmd = "stop listener -l ${lsnr_name} -n ${node} -f";
      my $run_as_owner = FALSE; # run as root
      my $status = srvctl($run_as_owner, $cmd, $crs_home);
      if (TRUE == $status)
      {
        trace("Successfully stopped listener '$lsnr_name' on node '$node'");
      }
      else
      {
        die(dieformat(180,
              "srvctl stop listener -l $lsnr_name -n $node -f", "null"));
      }
    }
  }
}

=head1 EXPORTED FUNCTIONS

=head2 startNetLsnr 
  
  Start the given node listener
 
=head3 Parameters
  
  [0] Whether or not to start the listener on current node
        TRUE  - on current node
        FALSE - on all nodes
  [1] Listener user name   
  [2] Listener name. When not specified, it defaults to "LISTENER" 
  
=head3 Returns
    
  None 
    
=cut

sub startNetLsnr
{
  my $local_node   = $_[0];
  my $lsnr_usrname = $_[1];
  my $lsnr_name    = $_[2];

  if ($local_node && isRimNode())
  {
    trace("Bypass startig node listener on a leaf node");
    return;
  }

  $lsnr_name = ($lsnr_name) ? ($lsnr_name) : "LISTENER";
  trace("Start node listener '$lsnr_name' with user name '$lsnr_usrname'");

  my $cmd;
  my $host = $CFG->HOST;
  my $srvctlbin = crs_exec_path('srvctl');  
  if ($local_node)
  {
    $cmd = "${srvctlbin} start listener -l ${lsnr_name} -n $host";
  }
  else
  {
    $cmd = "${srvctlbin} start listener -l ${lsnr_name}";
  }
  my $status = run_as_user($lsnr_usrname, ${cmd});

  if ((0 == $status) || (2 == $status))
  {
    trace("Successfully started node listener '$lsnr_name'");
  }
  else
  {
    die(dieformat(180,
          "srvctl start listener -l ${lsnr_name}", $status));
  }
}

=head1 EXPORTED FUNCTIONS

=head2 startListeners
  
  Start all local listeners
 
=head3 Parameters
  
  [0] Whether or not to start listeners on current node
        TRUE  - on current node
        FALSE - on all nodes
  
=head3 Returns
    
  TRUE or FALSE 
    
=cut

sub startListeners
{
  my $local_node = $_[0];

  if ($local_node && isRimNode())
  {
    trace("Bypass startig local listeneris on a leaf node");
    return TRUE;
  }
  
  my $success = TRUE;
  my (@capout, @cmd, $status);
  my $srvctl =  catfile($CFG->ORA_CRS_HOME, "bin", "srvctl");
  my $host   = $CFG->HOST;

  if ($local_node)
  {
    @cmd = ($srvctl, 'start', 'listener', '-n', $host);
  }
  else
  {
    @cmd = ($srvctl, 'start', 'listener');
  }
  $status = run_as_user2(getLsnrUsername(), \@capout, @cmd);

  if ((0 == $status) || (2 == $status))
  {
    trace ("@cmd ... success");
  }
  else
  {
    print_error(180, join(' ', @cmd), $status);
    $success = FALSE;
  }

  return $success;
}

=head1 EXPORTED FUNCTIONS

=head2 removeNetLsnr 
  
  Remove the current network listener
 
=head3 Parameters
  
  [0] GI home
  [1] Listener name. When not specified, it defaults to "LISTENER" 
  
=head3 Returns
    
  None 
    
=cut

sub removeNetLsnr
{
  my $crs_home = $_[0];
  my $name = $_[1];

  $name = ($name) ? ($name) : "LISTENER";
  trace("Remove Net Listener [$name]");

  my $lsnr_name;
  if (isOldVersionLT112())
  {
    my $rc;
    my @output;

    my @clunodes = split(',', $CFG->params('NODE_NAME_LIST'));
    foreach my $node (@clunodes)
    {
      $lsnr_name = $name."_".uc($node);
      trace("Created pre-11.2 style listener name: $lsnr_name");

      if (isOldVersionLT111())
      {
        my $crs_unregister = catfile($crs_home, 'bin', 'crs_unregister');
        my $resName = "ora\.".$node."\.".$lsnr_name."\.lsnr";
        @output = system_cmd_capture($crs_unregister, $resName);
        $rc = shift(@output);
        if ((0 == $rc) || (scalar(grep(/CRS-2613/, @output)) > 0) ||
                      (scalar(grep(/CRS-0210/, @output)) > 0))
        {
          trace("Successfully removed listener resource '$resName'");
        }
        else
        {
          die(dieformat(180, "crs_unregister $resName", $rc));
        }
      }
      else
      {
        my $srvctl = catfile($crs_home, 'bin', 'srvctl');
        @output = system_cmd_capture($srvctl,
                    'remove', 'listener', '-n', $node, '-l', $lsnr_name);
        $rc = shift(@output);
        if ((0 == $rc) || (2 == $rc))
        {
          trace("remove listener -n $node -l $lsnr_name ... passed");
        }    
        else
        {
          die(dieformat(180,
                 "srvctl remove listener -n $node -l $lsnr_name", $rc));
        }
      }
    }

    return;
  }

  $lsnr_name = $name;
  my $run_as_owner = FALSE; # run as root 
  my $status = srvctl($run_as_owner,
                       "remove listener -l ${lsnr_name} -f", $crs_home);
  if (TRUE == $status)
  {
    trace("remove listener -l $lsnr_name ... passed");
  }
  else
  {
    die(dieformat(180,
          "srvctl remove listener -l $lsnr_name", $status));
  }
}

=head1 EXPORTED FUNCTIONS

=head2 getCurrentNetLsnrs
  
  The subroutine for retrieving all existing listeners from a given CRS home
 
=head3 Parameters
  
  [0] GI home
  
=head3 Returns
    
  A hash array containing all existing listeners and listener homes
    
=cut

sub getCurrentNetLsnrs
{
  my $crs_home = $_[0];
  my %netLsnrs;
 
  trace("Retrieve all existing listeners from crs home: $crs_home");

  my $rc;
  my @output;
  my $srvctl = catfile($crs_home, 'bin', 'srvctl');

  if (isOldVersionLT112())
  { 
    my @clunodes = split(',', $CFG->params('NODE_NAME_LIST'));
    foreach my $node (@clunodes)
    {
      my $env = $ENV{'SRVM_TRACE'};
      undef $SRVM_TRACE;
      @output = system_cmd_capture($srvctl, 'config', 'listener', '-n', $node); 
      $ENV{'SRVM_TRACE'} = $env;
      $rc = shift(@output);
      if (0 == $rc)
      {
        foreach my $line (@output)
        {
          my $nodeName = (split(/\s+/, $line))[0];
          my $lsnrName = (split(/\s+/, $line))[1];
          my $ucNodeName = uc($nodeName);
          my $lsnrBase = $lsnrName;
          $lsnrBase =~ s/[_]$ucNodeName$//;

          if (isCkptPropertyExists("ROOTCRS_NODECONFIG", $lsnrBase))
          {
            trace("Listener '$lsnrBase' has been upgraded, hence bypassing query ...");
            next;
          }

          if (! defined($netLsnrs{$lsnrBase}))
          {
            $netLsnrs{$lsnrBase} = getLsnrHome($crs_home, $nodeName, $lsnrName);
          }
        }
      }
      else
      {
        die(dieformat(180, "srvctl config listener -n $node", $rc));
      }
    }
  }
  else
  {
    @output = system_cmd_capture($srvctl, 'config', 'listener');
    $rc = shift(@output);
    if (0 == $rc)
    {
      foreach my $line (@output)
      {
        if ($line =~ /Name/)
        {
          my $lsnrName = (split(/\s+/, $line))[1];
          $netLsnrs{$lsnrName} = getLsnrHome($crs_home, $CFG->HOST, $lsnrName); 
        }
      }
    }
    else
    {
      if (scalar(grep(/PRCN-2044/, @output)) > 0)
      {
        trace("No listener exists");
      }
      else
      { 
        die(dieformat(180, "srvctl config listener", $rc));
      }
    }
  }

  trace("All existing listeners: []");
  return %netLsnrs;
}

sub getLsnrHome
{
  my $crs_home  = $_[0];
  my $node_name = $_[1];
  my $lsnr_name = $_[2];
  my $lsnr_home;
  my $rc;
  my @output;
  my $resName;

  $lsnr_name = ($lsnr_name) ? ($lsnr_name) : "LISTENER";
  trace("Listener name: $lsnr_name");

  if (isOldVersionLT112())
  {
    $resName = "ora\.".$node_name."\.$lsnr_name"."\.lsnr";
    trace("Pre-11.2 listener resource name: $resName");

    my $crs_stat = catfile($crs_home, 'bin', 'crs_stat');
    @output = system_cmd_capture($crs_stat, '-p', $resName);  
    $rc     = shift(@output);
    if (0 != $rc)
    {
      die(dieformat(180, "crs_stat -p $resName", $rc));
    }
  }
  else
  {
    $resName = "ora\.".$lsnr_name."\.lsnr"; 
    trace("Listener resource name: $resName");
   
    my $crsctl =  catfile($crs_home, 'bin', 'crsctl');
    @output = system_cmd_capture($crsctl, "stat", "res", $resName, "-p");
    $rc     = shift(@output);
    if (0 != $rc)
    {
      die(dieformat(180, "crsctl stat res $resName -p", $rc));
    }
  }

  foreach my $line (@output)
  {
    if ($line =~ /ACTION_SCRIPT/)
    {
      my $actionScript = (split(/=/, $line))[1];
      $lsnr_home = (split(/bin/, $actionScript))[0];
      trace("Listener home = $lsnr_home");
      chop($lsnr_home);
      trace("Listener home after chopping: $lsnr_home");
      last;
    }
  }

  if ($lsnr_home =~ /\%CRS_HOME\%/)
  {
    $lsnr_home = $CFG->OLD_CRS_HOME;
  }

  return $lsnr_home;
}

=head1 EXPORTED FUNCTIONS

=head2 upgradeNetLsnrWithUsername
  
  Upgrade a network listener with given user name
 
=head3 Parameters
  
  [0] The owner of listener
  [1] Listener name. When not specified, it defaults to "LISTENER" 
  
=head3 Returns
    
  None 
    
=cut

sub upgradeNetLsnrWithUsername
{
  my $lsnr_username = $_[0];
  my $lsnr_home = $_[1];
  my $lsnr_name = $_[2];
  my $status;
  my $run_as_owner;

  $lsnr_name = ($lsnr_name) ? ($lsnr_name) : "LISTENER";

  if ($lsnr_username ne ($CFG->params('ORACLE_OWNER')))
  {
     trace("Upgrade listener '$lsnr_name' with username '$lsnr_username'");
     trace("Listener home: $lsnr_home");

     $run_as_owner = FALSE; # run as root
     $status = srvctl($run_as_owner,
          "add listener -l ${lsnr_name} -oraclehome ${lsnr_home} -user ${lsnr_username} -upgrade");
  }
  else
  {
     trace("Upgrade listener '$lsnr_name'");
     trace("Listener home: $lsnr_home");

     $run_as_owner = TRUE; # run as owner
     $status = srvctl($run_as_owner,
          "add listener -l ${lsnr_name} -oraclehome ${lsnr_home} -upgrade");
  }

  my $traceuser = ($lsnr_username ne ($CFG->params('ORACLE_OWNER'))) ? "-u $lsnr_username" : "";

  if (TRUE == $status)
  {
      trace("add listener -l $lsnr_name -oraclehome $lsnr_home" . "$traceuser" . " -upgrade ... passed");
  }
  else
  {
      die(dieformat(180,
            "srvctl add listener -l $lsnr_name -oraclehome $lsnr_home" . " $traceuser" . " -upgrade", $status));
  }
}


1;
